(ns metaheuristics.annealing)

(defprotocol AnnealingProblem
  "A protocol that specifies the inputs to the annealing
  function."
  (temperature [this iteration] "Returns the temperature that we reach for the input iteration. Iteration increases by one
  for each 'time unit'.")
  (successor [this current-solution] "Returns a valid successor solution that can be reached from the current solution")
  (cost [this solution] "Returns the cost of the provided solution."))

(defn- shake
  "Helper function which simulates 'shaking' in simulated annealing."
  [current-solution candidate delta temperature]
  (let [threshold (Math/pow (Math/E) (float (/ delta temperature)))]
    (if (> (rand) threshold)
      current-solution
      candidate)))

(defn anneal-seq
  "Returns a lazy sequence which iterates over solutions until we've 'cooled'. The starting state will be the
  first item returned in the sequence. The last item returned is the first state for which temp is less than
  or equal to 0."
  ([problem initial-state] (anneal-seq problem initial-state 0))
  ([problem current-solution iteration]
   (let [current-temp (temperature problem iteration)]
     (if (<= current-temp 0)
       (list current-solution)
       (cons current-solution
             (lazy-seq (let [next-candidate (successor problem current-solution)
                             delta (- (cost problem next-candidate) (cost problem current-solution))
                             next-solution (if (pos? delta)
                                             next-candidate
                                             (shake current-solution next-candidate delta current-temp))]
                         (anneal-seq problem next-solution (inc iteration)))))))))

(defn anneal
  "Attempts to find an optimal solution to the provided AnnealingProblem."
  ([problem initial-state] (anneal problem initial-state 0))
  ([problem current-solution iteration]
   (last (anneal-seq problem current-solution iteration))))