(ns madstap.comfy2
  (:refer-clojure :exclude [for doseq])
  #?(:cljs (:require-macros [madstap.comfy2 :refer [for forv]]))
  (:require
   [clojure.string :as str]
   [clojure.spec.alpha :as s]
   #?(:clj [clojure.edn :as edn])
   #?(:clj [clojure.core.specs.alpha])))

;;;;;;;;;;;;;;;;;;;;;
;; Variations on for

#?(:clj
   (s/def ::seq-expr-modifier
     (s/alt :let       (s/cat :k #{:let}
                              :bindings (s/and vector? :clojure.core.specs.alpha/bindings))
            :when-let  (s/cat :k #{:when-let}
                              :bindings :clojure.core.specs.alpha/bindings)
            :while-let (s/cat :k #{:while-let}
                              :bindings :clojure.core.specs.alpha/bindings)
            :do        (s/cat :k #{:do}        :expr any?)
            :when      (s/cat :k #{:when}      :expr any?)
            :when-not  (s/cat :k #{:when-not}  :expr any?)
            :while     (s/cat :k #{:while}     :expr any?)
            :while-not (s/cat :k #{:while-not} :expr any?))))

#?(:clj
   (s/def ::seq-exprs
     (s/and vector?
            (s/+ (s/cat :binding :clojure.core.specs.alpha/binding
                        :modifier (s/* ::seq-expr-modifier))))))

(defn join-seqs
  "Lazily concatenates a sequence-of-sequences into a flat sequence."
  {:no-doc true}
  [seq-of-seqs]
  (lazy-seq
   (when-let [s (seq seq-of-seqs)]
     (concat (first s) (join-seqs (rest s))))))

(defn- expand-*-let
  {:no-doc true}
  [kw bindings]
  (mapcat (fn [[binding expr]]
            `[:let [temp# ~expr] ~kw temp# :let [~binding temp#]])
          (partition 2 bindings)))

(defn- parse-expr
  {:no-doc true}
  [[k v :as seq-expr]]
  (case k
    :do [:let [(gensym) v]]
    :when-not  [:when  `(not ~v)]
    :while-not [:while `(not ~v)]
    :when-let (expand-*-let :when v)
    :while-let (expand-*-let :while v)
    seq-expr))

(defn parse-exprs
  {:no-doc true}
  [exprs]
  (into [] (mapcat parse-expr) (partition 2 exprs)))

#?(:clj
   (s/fdef for
     :args (s/cat :seq-exprs ::seq-exprs, :expr any?)))

(defmacro for
  "Like core/for, but adds :do, :when-let and :while-let modifiers.

  A drop-in replacement for core/for."
  {:style/indent 1, :added "1.0.1"}
  [seq-exprs expr]
  `(clojure.core/for ~(parse-exprs seq-exprs) ~expr))

#?(:clj
   (s/fdef forcat
     :args (s/cat :seq-exprs ::seq-exprs, :expr any?)))

(defmacro forcat
  "A for variant that presumes the body evaluates to a seqable thing.
  Returns a lazy sequence of all elements in each body."
  {:style/indent 1, :added "0.1.2"}
  [seq-exprs expr]
  `(join-seqs (for ~seq-exprs ~expr)))

#?(:clj
   (s/fdef forv
     :args (s/cat :seq-exprs ::seq-exprs, :body any?)))

(defmacro forv
  "Like for, but returns a vector. Not lazy."
  {:style/indent 1, :added "0.1.0"}
  [seq-exprs body-expr]
  `(vec (for ~seq-exprs ~body-expr)))

#?(:clj
   (s/fdef for-map
     :args (s/cat :seq-exprs ::seq-exprs, :key-expr any?, :val-expr any?)))

(defmacro for-map
  "Like for, but takes a key and value expression and returns a map.
  Multiple equal keys are treated as if by repeated assoc (last one wins).
  Not lazy."
  {:style/indent 1, :added "0.1.0"}
  [seq-exprs key-expr val-expr]
  `(into {} (for ~seq-exprs [~key-expr ~val-expr])))

#?(:clj
   (s/fdef forcatv
     :args (s/cat :seq-exprs ::seq-exprs, :body-expr any?)))

(defmacro forcatv
  "Like for, but presumes that the body-expr evaluates to a seqable thing,
  and returns a vector of every element from each body. Not lazy."
  {:style/indent 1, :added "0.2.0"}
  [seq-exprs body-expr]
  `(into [] cat (for ~seq-exprs ~body-expr)))

#?(:clj
   (s/fdef doseq
     :args (s/cat :seq-exprs ::seq-exprs, :body (s/* any?))))

(defmacro doseq
  "Like core/doseq, but adds :do, :when-let and :while-let modifiers.
  A drop-in replacement for core/doseq."
  [seq-exprs & body]
  `(clojure.core/doseq ~(parse-exprs seq-exprs) ~@body))
