;; This Source Code Form is subject to the terms of the Mozilla Public
;; License, v. 2.0. If a copy of the MPL was not distributed with this
;; file, You can obtain one at http://mozilla.org/MPL/2.0/

(ns magnet.duct.stork
  (:require [magnetcoop.stork :as stork]
            [integrant.core :as ig]
            [duct.logger :refer [log]]
            [datomic.api :as d]))

(defn get-all-migrations
  "Returns a vector of migrations from the `migration-files` collection.
  The collection must contain paths to resource files in .edn format.
  Logs to `logger` all the migration files that cannot be opened or
  are not valid .edn files. Those files are not part of the returned
  vector."
  [migration-files logger]
  (->> migration-files
       (mapv (fn [migration-file]
               (try
                 (stork/read-resource migration-file)
                 (catch Exception e
                   (log logger :error ::invalid-edn-file migration-file)
                   nil))))
       (filterv (complement nil?))))

(defn ensure-all-install!
  "Ensures that all provided `migrations` are installed.
  `conn` must be a valid Datomic connection. Logs to `logger` all the
  migrations that cannot be installed.
  "
  [conn migrations logger]
  (doseq [migration migrations]
    (try
      (stork/ensure-installed conn migration)
      (catch AssertionError e
        (log logger :error ::invalid-migration migration)))))

(defn migrations-status
  "Get the status of each of the provided `migrations`.
  `conn` must be a valid Datomic connection. Returns a map with the
  following structure:

  * keys are the migrations' :id values
  * values are the migrations' statuses (true if installed, false
    otherwise)"
  [conn migrations]
  (->>
   migrations
   ;; There might be syntactically valid .edn files that are not valid
   ;; migration maps. E.g., they may lack the :id key. In that case,
   ;; we want to filter those out from the returned map (as `{nil
   ;; false}` migration status is not very informative).
   (filter (fn [m] (:id m)))
   (map (fn [{:keys [id]}]
          [id (stork/installed? (d/db conn) id)]))
   (into {})))

(defmethod ig/init-key :magnet.duct/stork [_ {:keys [db-conn logger migration-files]}]
  (let [conn (:conn db-conn)
        migrations (get-all-migrations migration-files logger)
        status-before (migrations-status conn migrations)]
    (log logger :report ::found-migrations migrations)
    (log logger :report ::migrations-installed-status-before status-before)
    (ensure-all-install! conn migrations logger)
    (let [status-after (migrations-status conn migrations)]
      (log logger :report ::migrations-installed-status-after status-after)
      status-after)))
