(ns lazy-map.iop
  "Turn arbitrary Java objects into lazy maps, see [[extend-lazy-map]]."
  {:author "Artur Malabarba"}
  (:require [clojure.string :as s]
            [clojure.reflect :as reflect])
  (:require [lazy-map.core :refer [lazy-map]]))

(defn symbol-to-keyword
  "Convert a method or field name to a keyword.
  If the name starts with \"is\", the \"is\" is removed and a \"?\" is
  appended to the end. If the name starts with \"get\", the \"get\" is
  removed.
  
  Examples:
      toString :to-string
      isPerson :person?
      hasKindness :has-kindness
      getPrice :price"
  [method]
  (let [n (str method)
        rx (if (= n (s/upper-case n)) #"_" #"(?=[A-Z]+)")
        [head & tail :as s] (map s/lower-case (s/split n rx))]
    (keyword
     (str (s/join "-" (if (#{"is" "get"} head) tail s))
          (if (#{"is" "uses" "has"} head) "?")))))

(defn- public-methods
  [class]
  (->> (reflect/type-reflect class :ancestors true)
       (:members)
       (remove (partial instance? clojure.reflect.Constructor))
       (remove #(:static (:flags %)))
       (filter #(:public (:flags %)))))

(defn enum-to-keyword
  "Return a keyword representation of an Enum object.
  Lowercase the Enum's name and replace _ with -."
  [^java.lang.Enum enum]
  {:pre [(instance? java.lang.Enum enum)]
   :post [(keyword? %)]}
  (-> (.toString enum)
      (s/lower-case)
      (s/replace "_" "-")
      (keyword)))

;;; The protocol
(defprotocol ToLazyMap
  "Turn argument to a LazyMap. 
  Just Return it if already a map (lazy or otherwise)."
  (to-lazy-map [o] "Convert a Java object to a lazy map, where each entry corresponds to a method call."))

(extend-protocol ToLazyMap
  clojure.lang.Associative
  (to-lazy-map [a] a))

;;; The Macro
(defmacro extend-lazy-map
  "Extend [[to-lazy-map]] for converting an object of `class`.
  This macro must be used once on the class name, such as:
     `(extend-lazy-map String)`. 
  
  From then on, the `to-lazy-map` function can be used to convert an
  object of that class to a lazy map, where each entry correponds to a
  method call. Since everything is lazy, you can rest assured the
  methods won’t actually be called until you use them. The map keys
  are keywords generated from the method names with [[symbol-to-keyword]].

  For instance:

  ```clojure
  user> (to-lazy-map \"My Own Map!\")
  {:to-char-array #<LazyVal lazy_map.core.LazyVal@7da41d83>,
   :empty?        #<LazyVal lazy_map.core.LazyVal@46e7f2eb>,
   :to-string     #<LazyVal lazy_map.core.LazyVal@1b155784>,
   :intern        #<LazyVal lazy_map.core.LazyVal@63f1f826>,
   :chars         #<LazyVal lazy_map.core.LazyVal@3e64b9de>,
   :class         #<LazyVal lazy_map.core.LazyVal@47ed7453>,
   :length        #<LazyVal lazy_map.core.LazyVal@76f5a25d>,
   :trim          #<LazyVal lazy_map.core.LazyVal@1a29446d>,
   :bytes         #<LazyVal lazy_map.core.LazyVal@118a4bf>,
   :code-points   #<LazyVal lazy_map.core.LazyVal@2d7b1723>,
   :to-lower-case #<LazyVal lazy_map.core.LazyVal@76c7af9b>,
   :hash-code     #<LazyVal lazy_map.core.LazyVal@269d9702>,
   :object        #<LazyVal lazy_map.core.LazyVal@62e4e232>,
   :to-upper-case #<LazyVal lazy_map.core.LazyVal@17533342>}

  user> (:to-upper-case (to-lazy-map \"My Own Map!\"))
  \"MY OWN MAP!\"
  ```

  This macro also accepts a number of keyword arguments:

  - `exclude` is a sequence of methods to avoid (methods with non-zero
    arity or which return void are automatically avoided).
  - `post-fns` is a map from method names to functions. The function is
    called on the return value of that method. If the method's return
    value is an Enum, then this defaults to [[symbol-to-keyword]].
  - `keyname` is a map from method names to keys, these keys are used in
    place of the one generated by [[symbol-to-keyword]].
  - `allow-impure` is a boolean. By default, methods which return void
    are automatically avoided. Setting this to true disables that
    precaution.
  - `catch-exceptions` is a boolean which indicates to catch and discard
    any exceptions thrown by the methods (and return nil)."
  [class & {:keys [exclude post-fns keyname allow-impure catch-exceptions]}]
  (let [post (into {} post-fns)
        exc  (into #{} exclude)
        kn   (into {} keyname)
        arg  (vary-meta 'o assoc :tag class)
        class (resolve class)]
    `(extend-protocol ToLazyMap
       ~class
       (to-lazy-map [~arg]         
         (lazy-map
          ~(cond->> (public-methods class)
             true          (remove #(exc (:name %)))
             ;; no-exceptions (remove #(seq (:exception-types %)))
             (not allow-impure) (remove #(= (:return-type %) 'void))
             true          (filter #(= (count (:parameter-types %)) 0))
             true          (map (fn [{:keys [name return-type] :as o}]
                                  [(or (kn name) (symbol-to-keyword name))
                                   (if-let [f (or (post name)
                                                  (if (try (isa? (resolve return-type) Enum)
                                                           (catch Exception e))
                                                    `enum-to-keyword))]
                                     `(~f (. ~arg ~name))
                                     `(. ~arg ~name))]))
             catch-exceptions (map (fn [[k v]] [k `(try ~v (catch Exception ~'e))]))
             true          (into {:object arg})))))))

;;; And some extensions
(extend-lazy-map String)
(extend-lazy-map Exception)
