(ns manenko.clj-jira.core
  "Functions that wrap Jira API.

  ### Design

  Every Jira function in this namespace follows the same pattern and
  accepts at least the following three parameters: `send-fn`,
  `receive-fn`, and `opts`.

  * `send-fn`. A function of one argument, that accepts a [Ring-style]
    *request* map, sends it to a Jira HTTP server and returns
    a [Ring-style] *response* map.
  * `receive-fn`. A function of one argument, that accepts a [Ring-style]
    *response* map, transforms it and returns the *transformed* value.
  * `opts`. A map (it is optional for some of the functions) that configures
    a Jira query.

  The library doesn't send HTTP requests itself, delegating this job
  to a caller via `send-fn` and `receive-fn` parameters instead.  Each
  function will invoke `send-fn` argument with a request map it
  generated, then return a result of applying `receive-fn` to the
  return value from `send-fn`:

  ```clojure
  (defn typical-function
    [send-fn receive-fn opts]
    (let [request (generate-request-map opts)]
      (-> request
          send-fn
          receive-fn)))
  ```

  This approach allows for zero dependencies library but it comes with
  a price - users have to add a glue between HTTP library of their
  choice and `manenko/clj-jira`.

  ### Quickstart

  This section explains how to use `manenko/clj-jira` and [clj-http]
  libraries together.

  First, require both libraries in your application:

  ```clojure
  (ns manenko.clj-jira.example
    (:require [clj-http.client             :as client]
              [manenko.clj-jira.core       :as jira]
              [manenko.clj-jira.middleware :as middleware]))
  ```

  Then, define a host, email, and [API token] for Jira communication:

  ```clojure
  (def host  \"example.atlassian.net\")
  (def email \"user@example.com\")
  (def token \"DN21KLJh298haishu8AUHIU3\")
  ```

  You can, of course, retrieve them from other sources instead of hardcoding.

  Next step is to define a function that sends an HTTP request and
  integrate it with Jira middleware provided by `manenko/clj-jira`:

  ```clojure
  (defn request
    [m]
    (client/with-middleware
      (conj 
       client/default-middleware
       (middleware/wrap-api        host)
       (middleware/wrap-token-auth email token))
      (client/request m)))
  ```

  Now you can use `request` function to make API calls:
  
  ```clojure
  (jira/get-current-user request identity)
  ```

  [clj-http]: https://github.com/dakrone/clj-http
  [Ring-style]: https://github.com/ring-clojure/ring/blob/master/SPEC
  [API token]: https://confluence.atlassian.com/cloud/api-tokens-938839638.html"
  (:require [manenko.clj-jira.middleware :as middleware]
            [manenko.clj-jira.request    :as request]))

;; TODO: Expansion, pagination, and ordering
;; https://developer.atlassian.com/cloud/jira/platform/rest/v3/#expansion


(defn get-current-user
  "Gets information about a current user.

  **Arguments**

  **`send-fn`**
  
  A function of one argument, that accepts a Ring-style request map,
  sends it to a Jira HTTP server and returns a Ring-style response
  map.

  **`receive-fn`**
  
  A function of one argument, that accepts a Ring-style response map,
  transforms it and returns the transformed value.

  **`opts`**
  
  An optional map that configures a Jira query and supports the following keys:
  * `:expand` - A sequence of fields to include to the response.  Supports
                the following values: `:group` (returns all groups, including
                nested groups, the user belongs to) and `:applicationRoles`
                (returns the application roles the user is assigned to).

  The function uses [Get current user] Atlassian Jira API.

  [Get current user]: https://developer.atlassian.com/cloud/jira/platform/rest/v3/#api-rest-api-3-myself-get"
  ([send-fn receive-fn]
   (get-current-user send-fn receive-fn {}))
  ([send-fn receive-fn options]
   (let [request (request/get-current-user options)]
     (-> request
         send-fn
         receive-fn))))


(defn get-projects
  "Gets projects visible to a current user.

  **Arguments**

  * **`send-fn`**.
    A function of one argument, that accepts a Ring-style request map, sends it
    to a Jira HTTP server and returns a Ring-style response map.
  * **`receive-fn`**.
    A function of one argument, that accepts a Ring-style response map,
    transforms it and returns the transformed value.
  * **`opts`**.
    An optional map that configures a Jira query and supports the following
    keys:

      * `:startAt`.
        A page offset.  Integer.  Default is `0`.
      * `:maxResults`.
        A page size.  Integer.  Default is `0`.  Maximum is `0`.
      * `:orderBy`.
        Orders results by a field.  The following values are supported:
        `:category`, `:key`, `:name`, `:owner`.  Default is `:key`, which sorts
        projects alpabetically by project key.  You can prepend value with `+`
        or `-` to specify a sort direction, e.g. `:+owner`, `:-category`, etc.  
      * `:query`.
        Searches for projects which `key` or `name` matches the given string.
      * `:typeKey`.
        Orders results by the project type.  A sequence of the following values:
        `:business`, `:service_desk`, `:software`.
      * `:categoryId`.
        An identifier of the project's category.  Integer.
      * `:searchBy`.
        Default is `[:key, :name]`.
      * `:action`.
         Filters results by projects for which user can view, browse, or edit
         the project.  Supports the following values: `:view`, `:browse`,
         `:edit`.  Default is `view`.
      * `:expand`.
        A sequence of fields to include to the response.  Supports the following
        values: `:description`, `:projectKeys`, `:lead`, `:issueTypes`, `:url`.
  
  The function uses [Get projects paginated] Atlassian Jira API.

  [Get projects paginated]: https://developer.atlassian.com/cloud/jira/platform/rest/v3/#api-rest-api-3-project-search-get"
  ([send-fn receive-fn]
   (get-projects send-fn receive-fn {}))
  ([send-fn receive-fn opts]
   (let [request (request/get-projects opts)]
     (-> request
         send-fn
         receive-fn))))
