(ns manenko.clj-jira.middleware
  "Ring-style middleware for Jira REST API requests."
  (:require [clojure.string :as str]))


;; Middlware
;; -----------------------------------------------------------------------------

(defn ^:private create-api-url
  "Returns a full Jira REST API URL of the given `resource` on the
  given `host`."
  [host resource]
  (str "https://" host "/rest/api/3" resource))


;; Request headers
;; -----------------------------------------------------------------------------
;; - `x-atlassian-force-account-id=true``.  Operations will behave as if GDPR
;;                                          changes are enforced (for example,
;;                                          deprecated fields removed).  Use
;;                                          this header to test if your
;;                                          integration is GDPR-compliant.
;;

;; Response headers
;; -----------------------------------------------------------------------------
;; - `X-AACCOUNTID=id`.  This response header contains the Atlassian account ID
;;                       of the authenticated user.

(defn wrap-api
  [host]
  (fn [handler]
    (letfn [(resolve-url [request]
              (create-api-url host (::api-resource request)))
            (add-api-headers [request]
              (-> request
                  (assoc    :url (resolve-url request))
                  (assoc    :x-atlassian-force-account-id true)
                  (assoc-in [:headers "Content-Type"] "application/json")))]
      (fn
        ([request]
         (handler (add-api-headers request)))
        ([request respond raise]
         (handler (add-api-headers request) respond raise))))))


(defn wrap-token-auth
  [email token]
  (fn [handler]
    (letfn [(add-auth-headers [request]
              (assoc request :basic-auth [email token]))]
      (fn
        ([request]
         (handler (add-auth-headers request)))
        ([request respond raise]
         (handler (add-auth-headers request) respond raise))))))
