(ns structural-typing.api.type-repo
  "The `TypeRepo` structure and its functions."
  (:require [structural-typing.mechanics.canonicalizing-types :as canon]
            [structural-typing.mechanics.compiling-types :as compile]
            [structural-typing.api.defaults :as default]
            [structural-typing.frob :as frob]))

(defprotocol TypeRepoLike
  (hold-type [type-repo type-signifier type-descriptions])
  (check-type [type-repo type-signifier candidate])
  (replace-success-handler [type-repo handler]
    "For this `type-repo`, handle candidates that typecheck successfully by 
     passing them to `handler` as the last step in [[checked]]..")
  (replace-error-handler [type-repo handler]
    "For this `type-repo`, pass [[oopsies]] generated by type failures to 
     `handler` as the last step in [[checked]].")
  (the-success-handler [type-repo])
  (the-error-handler [type-repo]))

(defrecord TypeRepo [success-handler error-handler]
    TypeRepoLike
    (hold-type [type-repo type-signifier type-descriptions]
      (let [canonicalized (apply canon/canonicalize
                                 (:canonicalized-type-descriptions type-repo)
                                 type-descriptions)
            compiled (compile/compile-type canonicalized)]
        (-> type-repo 
            (assoc-in [:original-type-descriptions type-signifier] type-descriptions)
            (assoc-in [:canonicalized-type-descriptions type-signifier] canonicalized)
            (assoc-in [:compiled-types type-signifier] compiled))))

    (check-type [type-repo type-signifier candidate]
      (if-let [checker (get-in type-repo [:compiled-types type-signifier])]
        (checker candidate)
        (frob/boom! "There is no type `%s`" type-signifier)))
    
    (replace-error-handler [type-repo f]
      (assoc type-repo :error-handler f))

    (replace-success-handler [type-repo f]
      (assoc type-repo :success-handler f))

    (the-error-handler [type-repo] (:error-handler type-repo))
    (the-success-handler [type-repo] (:success-handler type-repo)))

(def empty-type-repo
  "A type repo that contains no types and uses the default success and error handlers."
  (->TypeRepo default/default-success-handler default/default-error-handler))

(defn origin
  "Returns the original description of the `type-signifier` (a sequence of vectors and maps)"
  [type-repo type-signifier]
  (get-in type-repo [:original-type-descriptions type-signifier]))

(defn description
  "Returns the canonical (expanded) description of the `type-signifier`."
  [type-repo type-signifier]
  (get-in type-repo [:canonicalized-type-descriptions type-signifier]))
