(ns clj-webdriver.remote.server
  (:use [clojure.java.io :only [as-url]]
        [clj-webdriver.driver :only [init-driver]]
        [clj-webdriver.core :only [get-url]]
        [clojure.tools.logging :as lg])
  (:require [clj-webdriver.util :as util])
  (:import clj_webdriver.ext.remote.RemoteWebDriverExt
           [org.mortbay.jetty Connector Server]
           org.mortbay.jetty.nio.SelectChannelConnector
           org.mortbay.jetty.security.SslSocketConnector
           org.mortbay.jetty.webapp.WebAppContext
           javax.servlet.Servlet
           org.openqa.selenium.remote.server.DriverServlet
           [org.openqa.selenium.remote
            DesiredCapabilities
            HttpCommandExecutor]))

(defprotocol IRemoteServer
  "Functions for managing a RemoteServer instance."
  (start [server] "Start the server. Will try to run stop if a bind exception occurs.")
  (stop [server] "Stop the server")
  (address [server] "Get address of the server")
  (new-remote-driver [server browser-spec] "Instantiate a new RemoteDriver record.")
  (start-remote-driver [server browser-spec target-url] "Start a new RemoteDriver record and go to `target-url`."))

(defn new-remote-webdriver*
  "Internal: wire up the `RemoteWebDriverExt` object correctly with a command executor and capabilities."
  ([remote-server browser-spec] (new-remote-webdriver* remote-server
                                                       browser-spec
                                                       {}
                                                       nil))
  ([remote-server browser-spec capabilities extra-capabilities]
   (lg/info remote-server browser-spec capabilities)
   (let [http-cmd-exec (HttpCommandExecutor. (as-url (address remote-server)))
         {:keys [browser]} browser-spec
         desired-caps (if (seq capabilities)
                        (DesiredCapabilities. (util/java-keys capabilities))
                        (util/call-method DesiredCapabilities browser nil nil))
         _ (when (seq extra-capabilities)
             (doseq [[prop-key prop-val] extra-capabilities]
               (.setCapability desired-caps prop-key prop-val)))
         _ (lg/info "creating remote driver:" http-cmd-exec)
         remote-webdriver (RemoteWebDriverExt. http-cmd-exec desired-caps)]
     (lg/info "created remote driver:")
     [remote-webdriver, desired-caps])))

(defrecord RemoteServer [connection-params webdriver-server]
  IRemoteServer
  (stop [remote-server]
    (.stop (:webdriver-server remote-server)))

  (start [remote-server]
    (try
      (let [port (get-in remote-server [:connection-params :port])
            path-spec (get-in remote-server [:connection-params :path-spec])
            server (Server.)
            context (WebAppContext.)
            connector (doto (SelectChannelConnector.)
                        (.setPort port))]
        (.setContextPath context "")
        (.setWar context ".")
        (.addHandler server context)
        (.addServlet context DriverServlet path-spec)
        (.addConnector server connector)
        (.start server)
        server)
      (catch java.net.BindException _
        (stop remote-server)
        (start remote-server))))

  (address [remote-server]
    (let [{:keys [host port path-spec existing]} (:connection-params remote-server)]
      (str "http://"
           host
           ":"
           port
           (clojure.string/join (drop-last path-spec))
           (when existing
             "hub"))))

  (new-remote-driver
    [remote-server browser-spec]
    (let [{:keys [browser profile capabilities cache-spec raw-capabilities]
           :or {browser :firefox cache-spec {}}} browser-spec
           ;; WebDriver object, DesiredCapabilities object based on capabilities
           ;; map passed in
           _ (lg/info "creating new remote webdriver")
           [webdriver desired-caps] (new-remote-webdriver* remote-server
                                                           {:browser browser
                                                            :profile profile}
                                                           capabilities
                                                           raw-capabilities)
           _ (lg/info "created new remote webdriver")
           ;; DesiredCapabilities as a Clojure map
           desired-capabilities (util/clojure-keys (into {} (.asMap desired-caps)))
           ;; actual capabilities (Java object) supported by the driver, despite your desired ones
           caps (.. webdriver getCapabilities)
           ;; actual capabilities as Clojure map, since once capabilities are
           ;; assigned to a driver you can't do anything but read them
           capabilities (util/clojure-keys (into {} (.asMap caps)))]
      (init-driver {:webdriver webdriver
                    :capabilities {:desired desired-capabilities
                                   :desired-obj desired-caps
                                   :actual capabilities
                                   :actual-obj caps}
                    :cache-spec cache-spec})))

  (start-remote-driver
    [remote-server browser-spec url]
    (let [driver (new-remote-driver remote-server browser-spec)]
      (get-url driver url)
      driver)))

(defn init-remote-server
  "Initialize a new RemoteServer record, optionally starting the
  server automatically (enabled by default)."
  [connection-params]
  (let [{:keys [host port path-spec existing] :or {host "127.0.0.1"
                                                   port 4444
                                                   path-spec "/wd/*"
                                                   existing false}} connection-params
                                                   server-record (RemoteServer. {:host host
                                                                                 :port port
                                                                                 :path-spec path-spec
                                                                                 :existing existing}
                                                                                nil)]
    (if (get-in server-record [:connection-params :existing])
      server-record
      (assoc server-record :webdriver-server (start server-record)))))

(defn remote-server?
  [rs]
  (:existing rs)
  ;(= (class rs) RemoteServer)
  )

(defn new-remote-session
  "Start up a server, start up a driver, return both in that
order. Pass a final falsey arg to prevent the server from being
started for you."
  ([] (new-remote-session {}))
  ([connection-params] (new-remote-session connection-params {:browser :firefox}))
  ([connection-params browser-spec]
     (let [new-server (init-remote-server connection-params)
           new-driver (new-remote-driver new-server browser-spec)]
       [new-server new-driver])))
