(ns materia.middleware
  (:require [clojure.string :as str]
            [clojurewerkz.gizmo.responder :refer [respond-with]]
            [clojurewerkz.gizmo.widget :as widget]
            [clojurewerkz.gizmo.request :as request]
            [materia.cljs.injector :refer [wrap-js-injector]]
            [materia.request :refer [wrap-request-binding]]
            [materia.services.db.core :refer [wrap-sql-logger]]
            [materia.utils :as u]
            [prone.middleware :refer [wrap-exceptions]]
            [ring.middleware.defaults :refer :all]
            [ring.middleware.refresh :refer [wrap-refresh]]
            [ring.middleware.reload :refer [wrap-reload]]
            [taoensso.timbre :as log]))

(defn wrap-request-logger [handler]
  (fn [req]
    (log/info (with-out-str (clojure.pprint/pprint req)))
    (handler req)))

(defn default-middleware [conf]
  #(wrap-defaults % conf))

(defn- normalize-header-key [s]
  (->> (str/split s #"-")
       (map str/capitalize)
       (str/join "-")))

(defn- normalize-headers [headers]
  (u/fmap-key normalize-header-key headers))

(defn wrap-header-normalizer [handler]
  (fn [req]
    (let [res (handler req)]
      (update-in res [:headers] normalize-headers))))

(defn- contain-meta? [ns ks]
  (some #(get-in (meta %) ks) (vals (ns-interns ns))))

(defn- contain-layout? [ns]
  (contain-meta? ns [:layout]))

(defn- contain-snippet? [ns]
  (contain-meta? ns [:snippet]))

(defn- find-ns-contains-templates []
  (->> (all-ns)
       (filter (some-fn contain-layout? contain-snippet?))
       (map ns-name)))

(defn wrap-template-reload [handler]
  (fn [req]
    (doseq [ns (find-ns-contains-templates)]
      (require ns :reload))
    (handler req)))


;; A monkey-patch for respond-with and wrap-responder. Original
;; implementation calls :html renderer even if :body key exists, if
;; its value is nil. It's not also a good idea to merge request and
;; response maps, because it could cause unintended disaster. e.g.,
;; You can never erase entries from session map.
(defn wrap-responder
  [handler]
  (fn [req]
    (let [res (handler req)
          res (if (and (contains? res :body)
                       (nil? (:body res)))
                (assoc res :body "")
                res)]
      (respond-with (assoc res :request req)))))

;; In this implementation, env-map will be passed to layout snippet
(defmethod respond-with :html
  [{:keys [widgets status headers layout] :as env}]
  (assert (> (count (widget/all-layouts)) 0) "Can't respond with :html without layouts given")
  (let [layout-template (if layout
                          (get (widget/all-layouts) layout)
                          (last (first (widget/all-layouts))))
        response        (request/with-request env
                          (widget/with-trace
                            (-> (layout-template env)
                                (widget/inject-core-widgets (:widgets env))
                                (widget/interpolate-widgets env)
                                widget/render*)))]
    {:headers (merge headers
                     {"content-type"  "text/html; charset=utf-8"
                      "content-length" (str (count response))})
     :status (or status 200)
     :body response}))

(defn wrap-ignore-trailing-slash
  "Modifies the request uri before calling the handler.
  Removes a single trailing slash from the end of the uri if present.

  Useful for handling optional trailing slashes until Compojure's route matching syntax supports regex.
  Adapted from http://stackoverflow.com/questions/8380468/compojure-regex-for-matching-a-trailing-slash"
  [handler]
  (fn [{uri :uri :as request}]
    (handler (assoc request :uri (cond (= "/" uri) uri
                                       (.endsWith uri "/") (apply str (butlast uri))
                                       :else uri)))))

(defn middlewares [{:keys [responder
                           request-logger
                           request-binding
                           auto-reload
                           auto-refresh
                           prone sql-logger
                           inject-js]
                    :as conf}]
  (u/conj-> []
            responder
            [0 wrap-responder]

            responder
            [10 wrap-header-normalizer]

            (and inject-js
                 (seq inject-js))
            [20 #(wrap-js-injector % inject-js)]

            request-logger
            [30 wrap-request-logger]

            request-binding
            [30 wrap-request-binding]

            auto-reload
            [100 wrap-reload]

            auto-reload
            [100 wrap-template-reload]

            auto-refresh
            [100 wrap-refresh]

            prone
            [200 wrap-exceptions]

            sql-logger
            [100 wrap-sql-logger]

            true
            [100 (default-middleware conf)]

            true
            [200 wrap-ignore-trailing-slash]))
