(ns materia.middleware.js-injector
  (:require [clojure.string :as str]
            [net.cgrand.enlive-html :as html]))

(defn- js-src [src]
  [:script {:type "text/javascript" :src src}])

(defn- js-require [ns]
  [:script {:type "text/javascript"}
   (format "goog.require('%s')" ns)])

(defn- inject-js [conf]
  (apply comp (concat (map (fn [ns]
                             (html/prepend (html/html (js-require ns))))
                           (get-in conf [:prepend :requires] []))
                      (map (fn [src]
                             (html/prepend (html/html (js-src src))))
                           (get-in conf [:prepend :paths] []))
                      (map (fn [ns]
                             (html/append (html/html (js-require ns))))
                           (get-in conf [:append :requires] []))
                      (map (fn [src]
                             (html/append (html/html (js-src src))))
                           (get-in conf [:append :paths] [])))))

(defn- normalize-body [body]
  (cond
    (string? body) (java.io.StringReader. body)
    (seq? body)    (java.io.StringReader. (str/join body))
    :else          body))

(defn- transform-html [body transformer]
  (-> (normalize-body body)
      html/html-resource
      (html/transform [:body] transformer)
      html/emit*))

(defn wrap-js-injector
  "Wraps js-injector middleware. `conf` is like as the following:

  {:prepend {:paths    [\"/js/out/goog/base.js\"]
  :requires []}
  :append {:paths []
  :requires [\"materia.dev\" \"materia.core\"]}}"
  [handler conf]
  (fn [req]
    (let [res (handler req)
          type (get-in res [:headers "Content-Type"] "text/html; charset=utf-8")]
      (if (and (re-find #"text/html" type)
               ((some-fn string? seq?) (:body res)))
        (update-in res [:body] transform-html (inject-js conf))
        res))))
