(ns materia.sql
  (:refer-clojure :exclude [update])
  (:require [materia.util.namespace :refer [import-ns]]
            [potemkin :refer [import-vars]]
            [stch.sql.format :as fmt]
            stch.sql.types
            [stch.sql.util :as su]))

(import-ns stch.sql)

(import-vars [stch.sql.types
              raw
              call])

(defprotocol ISQLFormat
  (sql-format [this opts]))

(extend-protocol ISQLFormat
  java.lang.String
  (sql-format [this opts] this)

  clojure.lang.IPersistentVector
  (sql-format [this opts] this)

  clojure.lang.IPersistentMap
  (sql-format [this opts]
    (apply fmt/format this (reduce into [] opts))))

(extend-protocol fmt/ToSQL
  clojure.lang.Cons
  (-to-sql [x]
    (fmt/-to-sql (apply list x))))

(defhelper merge-update [m tables]
  (update-in m [:update] concat (collify tables)))

(defmethod fmt/fn-handler "case-when" [_ test then else]
  (str "CASE WHEN " (fmt/to-sql test)
       " THEN "     (fmt/to-sql then)
       " ELSE "     (fmt/to-sql else)
       " END"))

(defmethod fmt/fn-handler "case" [_ & args]
  (str "CASE " (su/space-join (map fmt/to-sql args))
       " END"))

(defmethod fmt/fn-handler "when" [_ test then]
  (str "WHEN "  (fmt/to-sql test)
       " THEN " (fmt/to-sql then)))

(defmethod fmt/fn-handler "else" [_ else]
  (str "ELSE " (fmt/to-sql else)))

(defmethod fmt/fn-handler "inc" [_ x]
  (fmt/to-sql `(+ ~x 1)))

(defmethod fmt/fn-handler "dec" [_ x]
  (fmt/to-sql `(- ~x 1)))

(defn- expand-clause [c e]
  (if (= c :else)
    (list 'else e)
    (list 'when c e)))

(defmethod fmt/fn-handler "cond" [_ & clauses]
  "Clojure-ish version of SQL's case statement"
  (when clauses
    (assert (even? (count clauses)))
    (fmt/to-sql
     (cons 'case
           (->> (partition 2 clauses)
                (map #(apply expand-clause %)))))))
