(ns materia.routing
  (:require [bidi.bidi :as bidi]
            [cemerick.url :as url]
            [materia.handler :as h]
            [materia.response :as res]
            [materia.util :as u]
            [ring.util.request :as req]))

(defn endpoint [req]
  (get-in req [:materia/routing :handler]))

(defn create-router []
  (let [get-handler (h/create-handler-resolver)]
    (fn router [req]
      (if-let [handler (get-handler (endpoint req))]
        (handler req)
        (res/not-found!)))))

(defn create-dynamic-router []
  (fn router [req]
    (let [get-handler (h/create-handler-resolver)]
      (if-let [handler (get-handler (endpoint req))]
        (handler req)
        (res/not-found!)))))

(defn- inherit-route-params [req values keys]
  (reduce
   (fn [values' key]
     (if-let [v (or (get-in req [:route-params key])
                    (get-in req [:params key]))]
       (assoc values' key v)
       values'))
   values
   keys))

(defn- get-path [route route-id]
  (:path (first (filter #(= (:handler %) route-id) (bidi/route-seq route)))))

(defn- extract-route-params [pattern]
  (when (vector? pattern)
    (map bidi/param-key pattern)))

(defn- param-keys [route route-id]
  (->> route-id
       (get-path route)
       (map extract-route-params)
       (apply concat)
       (remove nil?)))

(defn href
  "Generate path + query string for the given route. If `inherit-keys`
  is specified, inherit parameters corresponding to each key from
  current request path. For example, let current path is /foo/page/20
  and the route definition is /:context/page/:page, then you can
  generate new path as follows using the parameter inheritance:

  (href req :route-id {:page 21} [:context]) ; => /foo/page/21"
  ([req route-id]
   (href req route-id {}))
  ([req route-id values]
   (href req route-id values []))
  ([req route-id values inherit-keys]
   (str
    (:context req)
    (let [route (u/maybe-deref (:materia/route req))
          params (inherit-route-params req values inherit-keys)
          query-params (apply dissoc params (param-keys route route-id))]
      (str (apply bidi/path-for route route-id (apply concat params))
           (when (seq query-params)
             (str "?" (url/map->query query-params))))))))

(defn match-request
  [route req]
  (apply bidi/match-route (u/maybe-deref route) (req/path-info req) (apply concat (seq req))))
