# Vault Clojure

Vault Clojure is a Clojure client for interacting with [HashiCorp Vault](https://www.vaultproject.io/) API.

## Installation

Library releases are pushed to Clojars. Add the following dependency to your leiningen's project definition:

```clojure
:dependencies [[mdl/vault-clojure "0.1.1"]]
```

[![Clojars Project](http://clojars.org/mdl/vault-clojure/latest-version.svg)](http://clojars.org/mdl/vault-clojure)


## Quick Start

Run vault in `dev` mode:

```
$ vault server -dev
```

Then on your `repl`,

```clojure
user=> (require '[vault.client :as vault])

user=> (def client (vault/http-client "http://localhost:8200"))

user=> (vault/authenticate client <the-root-token-from-vault>)

user=> (vault/write-secret client "secret/sample" {:hello "world"})
{:hello "world"}

user=> (vault/read-secret client "secret/sample")
Mar 09, 2016 5:39:37 PM vault.client invoke
INFO: Read secret/sample (valid for 2592000 seconds)
{:hello "world"}
```

## Usage

Require to load the library:

```clojure
(ns your.namespace
  (:require [vault.client :as vault]))
```

### Authenticate

In order to read and/or write secrets, you need to first authenticate the client. For now, only token based authentication is supported.

Create the client passing the url where vault api is listening:

```clojure
(def client (vault/http-client (System/getenv "VAULT_ADDR"))); read from $VAULT_ADDR variable
```

Then authenticate the client passing a valid token:

```clojure
(vault/authenticate client (System/getenv "VAULT_TOKEN")); read from $VAULT_TOKEN variable
```

Now, the client is ready to be used to read or write secrets into vault.

### Retrieve a secret

```clojure
(vault/read-secret client "secret/sample")
{:hello "world"}

```

### Create a secret

```clojure
(vault/write-secret client "secret/sample" {:goodbye "world"})
{:goodbye "world"}

```

### Complete example

The next example creates a small abstraction to be used from other parts of the system. It assumes that `VAULT_ADDR` and `VAULT_TOKEN` environment vars are passed to the system.

```clojure
(ns wallace.secrets.vault
  (:require [vault.client :as vault]))

;; assumes vault:8200 if not VAULT_ADDR env var
(def client (vault/http-client (or (System/getenv "VAULT_ADDR") "http://vault:8200")))

;; assumes a valid token within VAULT_TOKEN env.
(defn read-secret
  [path]
  (vault/authenticate client (System/getenv "VAULT_TOKEN"))
  (vault/read-secret client path))

```

then, from any other part of the same system:

```clojure
(ns wallace.server.components
  (:require [wallace.secrets.vault :as vault]))

; read db credentials from path passed within system's configuration
(defn fill-db-secrets
  [config]
  (let db-creds [(vault/read-secret (get-in config [:vault :db-creds-path]))]
  ...))
```

## License

Released under the [MIT License](http://www.opensource.org/licenses/MIT).

