(ns vault.client
  "For interacting with Hashicorp's Vault to read and write secrets using the HTTP API."
  (:require
    [clj-http.client :as http]
    [clojure.string :as str]
    [cheshire.core :as ch]))

;; ## Client Protocol

(defprotocol Client
  "Protocol for fetching secrets from Vault."

  (authenticate
    [client token]
    "Updates the client's internal state by authenticating with the given
    credentials.")

  (read-secret
    [client path]
    "Reads a secret from a specific path.")

  (write-secret
    [client path secret]
    "Write a secret to a specific path."))


;; ## HTTP API Client

(defrecord HTTPClient
  [api-url token]

  Client

  (authenticate
    [this client-token]
    (let [response (http/get (str api-url "/v1/auth/token/lookup-self")
                     {:headers {"X-Vault-Token" client-token}
                      :accept :json
                      :as :json})]
      (when-let [data (get-in response [:body :data])]
        (reset! token client-token))
      this))


  (read-secret
    [this path]
    (when-not (string? path)
      (throw (IllegalArgumentException.
               (str "Secret path must be a string, got: " (pr-str path)))))
    (when-not @token
      (throw (IllegalStateException.
               (str "Cannot read path " path " with unauthenticated client."))))
    (let [response (http/get (str api-url "/v1/" path)
                     {:headers {"X-Vault-Token" @token}
                      :accept :json
                      :as :json})]
      (get-in response [:body :data])))


  (write-secret
    [this path secrets]
    (when-not (string? path)
      (throw (IllegalArgumentException.
               (str "Secret path must be a string, got: " (pr-str path)))))
    (when-not (map? secrets)
      (throw (IllegalArgumentException.
               (str "Secrets must be a map, got: " (pr-str secrets)))))
    (when-not @token
      (throw (IllegalStateException.
               (str "Cannot read path " path " with unauthenticated client."))))
    (let [response (http/post (str api-url "/v1/" path)
                     {:headers {"X-Vault-Token" @token}
                      :body (ch/generate-string secrets)
                      :accept :json
                      :as :json})]
      secrets)))

;; Remove automatic constructors.
(ns-unmap *ns* '->HTTPClient)
(ns-unmap *ns* 'map->HTTPClient)


(defn http-client
  "Constructs a new HTTP Vault client."
  [api-url]
    (when-not (string? api-url)
      (throw (IllegalArgumentException.
               (str "Vault api-url must be a string, got: " (pr-str api-url)))))
    (HTTPClient. api-url (atom nil)))
