(ns toothpick.assembler
  (:require [toothpick.core :refer [bit-mask-n]]))


;; Subsystem for using ISA descriptors to assemble instruction descriptor
;; structures into bytes usable by a bytecode machine.
;; ------------------------------------------------------------------------------
(defn encode-field
  "Encodes an instruction field, using the field descriptor map and a map of
  parameter names to values. Returns an integer representing the encoded field."

  [icode field val-map]
  (case (:type field)
    ;; case for encoding a constant field... Note that this does _not_
    ;; make an effort to get a parameter from the parameters map,
    ;; because its a constant!
    (:const)
      (bit-shift-left
       (bit-and (bit-mask-n (:width field))
                (:value field))
       (:offset field))

    ;; Because this case is a parameter that has to be encoded in, a
    ;; parameter is fetched from the params map.
    (:unsigned-field :field :signed-field)
      (let [val (get val-map (:name field) 0)]
        (assert ((:pred field) val)
                (format "Failed to encode parameter %s" (name (:name field))))
        (bit-shift-left
         (bit-and (bit-mask-n (:width field))
                  val)
         (:offset field)))))


(defn map->bytecode
  "Compiles an instruction parameter map which the assembler would
  produce. Returns the integer encoding of the specified opcode with the given
  parameter values."

  [isa opcode val-map]
  (let [icode (get (:icodes isa) opcode)]
    (assert icode 
            (format "Could not get an icode for name %s" opcode))
    (let [fields (:fields icode)]
      (assert fields 
              (format "Could not get icode fields for icode %s"
                             (:name icode)))
      (let [encoding (mapv #(encode-field icode %1 val-map) fields)]
        (reduce bit-or 0 encoding)))))


(defn list->bytecode
  "Compiles a list that a programmer could actually type or generate
  into an assembled word."

  [isa [name & tail]]
  (let [opcode (get-in isa [:icodes name])]
    (assert opcode (format "Failed to find opcode in isa: %s" opcode))
    (let [{:keys [fields params] :as icode} opcode
          val-map (zipmap params tail)]
      (map->bytecode isa name val-map))))


;; Define a prototype assembler.
;;------------------------------------------------------------------------------
(defn label? [form]
  (or (and (list? form)
           (= (first form) :label)
           (keyword? (second form)))
      (keyword? form)))


(defn label-symbol 

  [form]
  (when (label? form)
    (cond (list? form)
            (second form)
          (keyword? form)
            form)))


(defn byte-count [form]
  4)


(defn compute-labels 
  
  [start forms]
  (loop [label-addr-map {}
         address start
         forms forms]
    (let [head (first forms)]
      (if (label? head)
        (recur (assoc label-addr-map (label-symbol head) address)
               address
               (rest forms))
        (if-not (empty? (rest forms))
          (recur label-addr-map
                 (+ address (byte-count head))
                 (rest forms))
          label-addr-map)))))


(defn resolve-param
  "Assumes that all explicit parameters are integers and that keywords are
  labels. Looks up keywords in the argument translation table and returns either
  the translation or the existing parameter. Assertion fails if the keyword is
  not found."

  [label-map param]
  (if (label? param)
    (do (assert (contains? label-map param) 
                (format "Label %s undefined!" param))
        (get label-map (label-symbol param)))
    param))


(defn resolve-params 
  "Resolves the parameters of an opcode, being the _tail_ of the opcode. The
  head element is ignored and assumed to be an instruction keyword. Returns a
  new sequence representing the resolved opcode."

  [label-map icode]
  (let [op (first icode)
        more (rest icode)]
    (cons op (map #(resolve-param label-map %1) more))))


(defn assemble
  "Compiles a series of assembler directives into a useable bytecode,
  computing relative jumps and absolute instruction positions naively.
  This is _not_ the final form of the assembler, only a prototype
  therefor which will be used to inform later syntax and design
  decisions."

  [isa forms & {:keys [start]
                :as   options
                :or   {start 0}}]
  (let [label-map (compute-labels start forms)]
    (as-> forms v
          (remove label? v)
          (map #(resolve-params label-map %1) v)
          (map #(list->bytecode isa %1) v))))
