(ns clj-infra.core
  (:require [cheshire.core :as json]
            [clojure.repl :as r]
            [clojure.string :as str]
            [prevayler-clj.prevayler4 :refer [prevayler! handle!]])
  (:import (java.io File FileWriter BufferedWriter PrintWriter)
           (java.time Instant LocalTime)
           (java.time.format DateTimeFormatter)
           (java.util.function Consumer)))

(def fmt (DateTimeFormatter/ofPattern "HH:mm:ss.SSS"))

(def initial-state {:version 0,
                    :rollbacks {}
                    :data {}})

(defn- to-symbol [f]
  (-> f str r/demunge (str/split #"@") first symbol))

(defn- delete-keys [m ks]
  (reduce (fn [m k] (dissoc m k)) m ks))

(def ^:dynamic *print* println)

(defn print-to-file-fn [^File log-file]
  (let [wtr (-> log-file
                FileWriter.
                BufferedWriter.
                (PrintWriter. true)
                agent)]
    (fn [msg]
      (letfn [(write [out msg] (.println out msg) out)]
        (send wtr write msg)))))

(defn echo
  ([say-now]
   (echo 2 say-now))
  ([level say-now]
   (let [msg (format "%s[%s] %s%s"
                     (if (= 0 level) "\n" "")
                     (.format fmt (LocalTime/now))
                     (str/join (repeat (* 2 level) " "))
                     say-now)]
     (*print* msg))
   nil)

  ;; Allows echo to be used as part of rollups/rollbacks
  ([_ _ _ level say-now] (echo level say-now))
  ([_ _ _ say-now] (echo say-now)))

(defn- rewire-fn [verbose? dry-run? presets]
  (fn [[f & rest]]
    (let [args (concat presets rest)]
      (when (or verbose? dry-run?)
        (echo (format "(apply %s %s)" f rest)))
      (when-not dry-run?
        (apply (if (symbol? f) (resolve f) f) args)))))

(defn- track-state [state
                    {:keys [version rollbacks restarts data delete-data]}
                    timestamp]
  (-> state
      (assoc :version version)
      ((fn [m] (if (seq rollbacks)
                 (assoc-in m [:rollbacks version]
                           [(-> timestamp (Instant/ofEpochMilli) str)
                            rollbacks])
                 m)))
      ((fn [m] (if (seq restarts)
                 (assoc-in m [:restarts version]
                           [(-> timestamp (Instant/ofEpochMilli) str)
                            restarts])
                 m)))
      ((fn [m] (if (seq data)
                 (update m :data merge data)
                 m)))
      ((fn [m] (if (seq delete-data)
                 (update m :data delete-keys delete-data)
                 m)))
      ((fn [m]
         (-> m
             (update
               :rollbacks
               (fn [v]
                 (apply dissoc v (filter (partial < version) (keys v)))))
             (update
               :restarts
               (fn [v]
                 (apply dissoc v (filter (partial < version) (keys v))))))))))

(defn secret-from-env [env k]
  (let [env-name (str/upper-case (format "%s_%s" env
                                         (-> k name (str/replace #"-" "_"))))
        v (System/getenv env-name)]
    (if v
      v
      (throw (IllegalStateException.
               (format
                 (str "Could not find value at env var '%s' "
                      "while attempting to retrieve secret '%s' in env '%s'")
                 env-name k env))))))

(defn apply-infra!
  ([p s envk dry-run? verbose? versions]
   (apply-infra! p s envk < dry-run? verbose? versions))
  ([p s envk include-version? dry-run? verbose? versions]
   (let [{:keys [version]} @p]
     (doseq [[new-version cmds] (partition 2 versions)]
       (when (include-version? version new-version)
         (echo 1 (format "Getting to version %s" new-version))
         (let [results
               (doall (map (rewire-fn verbose? dry-run? [p s envk]) cmds))

               rollbacks (map :rollback results)
               restarts (map :restart results)
               new-data (reduce merge (map :data results))
               data-to-delete (reduce concat (map :delete-data results))]
           (when (seq new-data)
             (echo (format "Adding data %s" (str/join ", " (keys new-data)))))
           (when (seq data-to-delete)
             (echo (format "Deleting data %s" (str/join ", " data-to-delete))))
           (when-not dry-run?
             (handle! p {:version new-version
                         :rollbacks (->> rollbacks
                                         (remove nil?)
                                         ((fn [rollbacks]
                                            (map (fn [[f & rest]]
                                                   (into [(to-symbol f)] rest))
                                                 rollbacks))))
                         :restarts (->> restarts
                                        (remove nil?)
                                        ((fn [restarts]
                                           (map (fn [[f & rest]]
                                                  (into [(to-symbol f)] rest))
                                                restarts))))
                         :data new-data
                         :delete-data data-to-delete}))
           {:version new-version}))))))

(defn rollback-infra! [p s envk dest-version dry-run? verbose?]
  (let [{:keys [version rollbacks]} @p

        rollbacks-to-apply
        (->> (range (inc dest-version) (inc version))
             sort
             reverse
             (map (juxt dec (comp second rollbacks)))
             (apply concat))]
    (apply-infra! p s envk >= dry-run? verbose? rollbacks-to-apply)))

(defn restart! [p s envk & extra-args]
  (let [restart-fns (:restarts @p)
        restart-versions (->> restart-fns keys sort reverse)
        presets (concat [p s envk] extra-args)]
    (doseq [restart restart-versions]
      (echo (format "Restarting %s" restart))
      (let [[_ cmds] (restart-fns restart)]
        (doall (map (rewire-fn false false presets) cmds))))))

(defn debug [p _s _envk & _args]
  (-> @p (json/generate-string {:pretty true}) println))

(defn with-infra-tracking! [^File db-file, ^Consumer receive-prevayler]
  (with-open [p (prevayler! {:initial-state initial-state
                             :business-fn track-state
                             :journal-file db-file})]
    (receive-prevayler p)))
