(ns clj-infra.pleroma
  (:require [cheshire.core :as json]
            [clj-infra.core :as infra]
            [clj-infra.digitalocean :as do]
            [clj-infra.shell :as sh]
            [clj-yaml.core :as yaml]
            [clojure.java.io :as io]
            [clojure.stacktrace :as st]
            [clojure.string :as str]))

;; This is an example of how to use the clj-infra library to set up
;; a Pleroma instance in DigitalOcean. There is a lot more that
;; you can configure after you've gotten things up and running, but
;; you will have a working app when this thing is all done.
;;
;; Check out
;; <https://docs-develop.pleroma.social/backend/configuration/cheatsheet/>
;; once you're up and running!
;;
;; Before you run this, you need to create a DigitalOcean account and set
;; up an api token. Then configure environment variables appropriately.
;; You can check out envrc.pleroma.example at the root of this project
;; for more. You can copy that file as `.envrc` and if you use
;; direnv it should work!
;; See <https://direnv.net/>
;;
;; Once you've got everything configured, just run `lein pleroma <env>`
;; where `<env>` is the name of the environment you want to be configured.
;; For example `lein pleroma prod`
;;
;; The env vars you have set up should all be prefixed with the name
;; of your env. For example, if you say `lein pleroma prod` you should
;; have env vars like `PROD_ROOT_DOMAIN`

(def region "nyc3")
(def pleroma-port 4000)
(def health-check-path "/api/v1/pleroma/healthcheck")

(defn show-usage []
  (println "Usage:"
           (str "lein run <env> "
                "[dest version | 'debug' | 'restart' | 'dryrun'] "
                "['dryrun' | 'verbose']")
           "\n"
           "Examples:\n\n"
           "lein run prod           # Applies all unapplied infra\n\n"
           "lein run prod dryrun    # Shows steps but doesn't do it\n\n"
           "lein run prod 0         # Rolls back all infra\n\n"
           "lein run prod 2         # Rolls back to version 2\n\n"
           "lein run prod 2 dryrun  # Shows steps but doesn't do it\n\n"
           "lein run prod 2 verbose # Shows steps and does it\n\n"
           "lein run prod debug     # Dump the entire local state\n\n"
           "lein run prod restart   # Reconfigure/restart components"))

(defn- env-tags [envk] [(format "env:%s" (name envk))])

(defn remove-ssh-public-key [_p s _envk key-name key-id]
  (infra/echo "Removing SSH public key")
  (do/remove-ssh-public-key (s :do-api-token) key-id)
  (infra/cleanup-data key-name))

(defn ssh-key-name [key-name] (->> key-name (format "ssh-pk-%s") keyword))

(defn ssh-key-id [data] (get-in data [:ssh_key :id]))

(defn upload-ssh-public-key [_p s _envk key-name]
  (infra/echo "Uploading SSH public key")
  (let [key-filename (s :do-ssh-key)
        key (slurp key-filename)

        result
        (do/upload-ssh-public-key (s :do-api-token) key-name key)

        key-id (ssh-key-id result)

        key-name (ssh-key-name key-name)]
    {:rollback [remove-ssh-public-key key-name key-id]
     :data {key-name result}}))

(defn dbc-key-name [key-name] (->> key-name (format "dbc-%s") keyword))

(defn dbc-id [data] (get-in data [:database :id]))

(defn delete-postgresql [_p s _envk key-name dbc-id]
  (infra/echo (format "Deleting postgresql database '%s'" key-name))
  (do/delete-postgresql-db-cluster (s :do-api-token) dbc-id)
  (infra/cleanup-data key-name))

(defn create-postgresql [_p s envk dbc-name]
  (infra/echo (format "Creating postgresql database '%s'" dbc-name))
  (let [key-name (dbc-key-name dbc-name)

        create-result
        (do/create-postgresql-db-cluster (s :do-api-token)
                                         region
                                         dbc-name
                                         "db-s-1vcpu-1gb"
                                         1
                                         (env-tags envk))]
    {:rollback [delete-postgresql key-name (dbc-id create-result)]
     :data {key-name create-result}}))

(defn wait-for-db [p s _envk dbc-name]
  (infra/echo (format "Waiting for db '%s' to be online" dbc-name))
  (let [k (dbc-key-name dbc-name)
        dbc-id (dbc-id (-> @p :data k))
        token (s :do-api-token)]
    (loop [current-status nil]
      (let [status (do/get-dbc-status token dbc-id)]
        (when-not (= status current-status)
          (infra/echo
            3
            (format "Status changed from %s to %s" current-status status)))
        (when-not (= "online" status)
          (Thread/sleep 3000)
          (recur status))))))

(defn db-key-name [dbc-name db-name]
  (keyword (format "db-%s-%s" dbc-name db-name)))

(defn delete-postgresql-db [_ _ _envk _ _ db-name]
  (infra/echo (format "Cowardly refusing to delete database '%s'" db-name)))

(defn add-postgresql-db [p s _envk dbc-name db-name]
  (infra/echo (format "Adding database '%s'" db-name))
  (let [k (dbc-key-name dbc-name)
        db-data (-> @p :data k)
        dbc-id (dbc-id db-data)
        token (s :do-api-token)
        db-key (db-key-name dbc-name db-name)]
    {:data {db-key (do/add-postgresql-db token dbc-id db-name)}
     :rollback [delete-postgresql-db db-key dbc-name db-name]}))

(defn- droplet-key [droplet-name]
  (->> droplet-name (format "droplet-%s") keyword))

(defn- droplet-id
  ([data] (get-in data [:droplet :id]))
  ([p droplet-name]
   (let [k (droplet-key droplet-name)]
     (droplet-id (-> @p :data k)))))

(defn- droplet-ip [p s droplet-name]
  (let [droplet-id (droplet-id p droplet-name)
        droplet-info (do/get-droplet-info (s :do-api-token) droplet-id)]
    (do/get-droplet-ip droplet-info)))

(defn destroy-droplet [_p s _envk droplet-id droplet-key]
  (infra/echo (format "Destroying droplet '%s'" droplet-key))
  (do/destroy-droplet (s :do-api-token) droplet-id)
  (infra/cleanup-data droplet-key))

(defn create-droplet [p s envk droplet-name size image key-name]
  (infra/echo (format "Creating droplet '%s': %s/%s with SSH key %s"
                      droplet-name size image key-name))
  (let [k (droplet-key droplet-name)
        ssh-key-name (ssh-key-name key-name)
        ssh-key-data (-> @p :data ssh-key-name)
        ssh-key-id (ssh-key-id ssh-key-data)
        result (do/create-simple-droplet (s :do-api-token)
                                         region
                                         droplet-name
                                         size
                                         image
                                         ssh-key-id
                                         (env-tags envk))
        droplet-id (droplet-id result)]
    {:rollback [destroy-droplet droplet-id k]
     :data {k result}}))

(defn wait-for-droplet [p s _envk droplet-name]
  (infra/echo (format "Waiting for droplet '%s' to be active" droplet-name))
  (let [droplet-id (droplet-id p droplet-name)
        token (s :do-api-token)]
    (loop [current-status nil]
      (let [status (do/get-droplet-status token droplet-id)]
        (when-not (= status current-status)
          (infra/echo
            3
            (format "Status changed from %s to %s" current-status status)))
        (when-not (= "active" status)
          (Thread/sleep 3000)
          (recur status)))))
  nil)

(defn- docker-compose [{:keys [domain
                               container-name
                               instance-name
                               admin-email
                               notify-email
                               smtp-server
                               smtp-username
                               smtp-password
                               db-host
                               db-port
                               db-name
                               db-user
                               db-password]}]
  (yaml/generate-string
    {:version "3.1",

     :services
     {:pleroma
      {:image "pleroma",
       :build {:context "."}
       :container_name container-name,
       :hostname "pleroma",
       :labels ["org.label-schema.group=pleroma"],
       :restart "always",

       :environment
       {:DB_USER db-user
        :DB_PASS db-password
        :DB_HOST db-host
        :DB_NAME db-name
        :DB_PORT db-port
        :INSTANCE_NAME instance-name
        :ADMIN_EMAIL admin-email
        :NOTIFY_EMAIL notify-email
        :SMTP_SERVER smtp-server
        :SMTP_USERNAME smtp-username
        :SMTP_PASSWORD smtp-password
        :DOMAIN domain
        :PORT pleroma-port},

       :ports [(format "%s:%s" pleroma-port pleroma-port)],

       :volumes
       ["./uploads:/var/lib/pleroma/uploads"
        "./static:/var/lib/pleroma/static"
        "./config.exs:/etc/pleroma/config.exs:ro"]},}}

    :dumper-options
    {:indent 2
     :indicator-indent 1
     :flow-style :block}))

(defn configure-pleroma [p s _envk config]
  (infra/echo "Configuring pleroma")
  (let [{:keys [dbc-name droplet-name db-name]} config
        dbc-key (dbc-key-name dbc-name)
        dbc-data (-> @p :data dbc-key)
        droplet-ip (droplet-ip p s droplet-name)
        {:keys [host port user password]} (-> dbc-data :database :connection)
        ssh-priv-key (s :do-ssh-priv-key)]
    (sh/upload-file
      ssh-priv-key
      droplet-ip
      "docker-compose.yml"
      (docker-compose
        (merge
          config
          {:instance-name (s :instance-name)
           :admin-email (s :admin-email)
           :notify-email (s :notify-email)
           :smtp-server (s :smtp-server)
           :smtp-username (s :smtp-username)
           :smtp-password (s :smtp-password)
           :db-host host
           :db-port port
           :db-name db-name
           :db-user user,
           :db-password password})))

    (sh/upload-file ssh-priv-key
                    droplet-ip
                    "Dockerfile"
                    (slurp "resources/pleroma/Dockerfile"))
    (sh/upload-file ssh-priv-key
                    droplet-ip
                    "config.exs"
                    (slurp "resources/pleroma/config.exs"))
    nil))

(defn clean-pleroma-docker [p s _envk droplet-name]
  (infra/echo "Cleaning pleroma docker build")
  (apply
    sh/exec
    (concat (sh/ssh-cmd (s :do-ssh-priv-key)
                        (droplet-ip p s droplet-name))
            ["docker" "system" "prune" "--all" "--force"])))

(defn build-pleroma [p s _envk droplet-name]
  (infra/echo "Building pleroma")
  (apply
    sh/exec
    (concat (sh/ssh-cmd (s :do-ssh-priv-key)
                        (droplet-ip p s droplet-name))
            ["docker-compose" "build"]))
  {:rollback [clean-pleroma-docker droplet-name]})

(defn stop-pleroma [p s _envk droplet-name]
  (infra/echo "Stopping pleroma")
  (apply
    sh/exec
    (concat (sh/ssh-cmd (s :do-ssh-priv-key)
                        (droplet-ip p s droplet-name))
            ["docker-compose" "down"])))

(defn start-pleroma [p s _envk droplet-name]
  (infra/echo "Starting pleroma")
  (infra/echo "This will take a WHILE (needs to build pleroma)")
  (apply
    sh/exec
    (concat (sh/ssh-cmd (s :do-ssh-priv-key)
                        (droplet-ip p s droplet-name))
            ["docker-compose" "up" "-d"]))
  {:rollback [stop-pleroma droplet-name]})

(defn wait-for-pleroma [p s _envk droplet-name]
  (infra/echo "Waiting for pleroma to become healthy")
  (loop [healthy false
         tries 0]
    (when-not healthy
      (when (> 0 tries) (Thread/sleep 3000))
      (let [new-healthy
            (try
              (-> (apply
                    sh/exec
                    (concat
                      (sh/ssh-cmd (s :do-ssh-priv-key)
                                  (droplet-ip p s droplet-name))
                      ["curl"
                       (format "http://localhost:4000%s"
                               health-check-path)]))
                  (json/parse-string keyword)
                  :healthy)
              (catch Exception e
                (infra/echo (.getMessage e))
                false))]
        (recur new-healthy (inc tries))))))

(defn add-pleroma-admin-user [p s _envk
                              droplet-name
                              container-name]
  (let [dip (droplet-ip p s droplet-name)
        add-user-script "add-admin-user.sh"
        admin-username (s :admin-username)
        admin-email (s :admin-email)]
    (infra/echo (format "Adding pleroma admin user '%s (%s)'"
                        admin-username admin-email))
    (sh/upload-file
      (s :do-ssh-priv-key)
      dip
      add-user-script
      (format (str "docker exec %s sh /pleroma/bin/pleroma_ctl user new "
                   "'%s' '%s' --admin --assume-yes")
              container-name admin-username admin-email))
    (apply sh/exec
           (concat (sh/ssh-cmd (s :do-ssh-priv-key) dip)
                   ["bash" add-user-script])))
  nil)

(defn name-from-domain [domain]
  (str/replace domain #"[.]" "-"))

(defn ssl-certk [cert-name]
  (-> (format "ssl-%s" cert-name) keyword))

(defn ssl-cert-id [p certk]
  (-> @p :data certk :certificate :id))

(defn remove-ssl-cert [p s _envk cert-name]
  (let [token (s :do-api-token)
        certk (ssl-certk cert-name)]
    (infra/echo (format "Deleting SSL Cert %s" cert-name))
    (do/delete-ssl-cert token (ssl-cert-id p certk))
    (infra/cleanup-data certk)))

(defn create-ssl-cert [_p s _envk domain]
  (let [token (s :do-api-token)
        cert-name (name-from-domain domain)
        certk (ssl-certk cert-name)

        result
        (do/create-ssl-cert token cert-name [domain])]
    (infra/echo (format "Created SSL Cert '%s'" cert-name))

    {:data {certk result}
     :rollback [remove-ssl-cert cert-name]}))

(defn lbk [lb-name]
  (-> (format "lb-%s" lb-name) keyword))

(defn remove-load-balancer [p s _envk lb-k]
  (let [token (s :do-api-token)
        id (-> @p :data lb-k :load_balancer :id)]
    (infra/echo (format "Deleting load balancer '%s'" id))
    (do/delete-load-balancer token id)
    (infra/cleanup-data lb-k)))

(defn create-load-balancer [p s _envk droplet-name domain]
  (infra/echo (format "Creating load balancer for '%s'->'%s'"
                      domain
                      droplet-name))
  (let [token (s :do-api-token)
        did (droplet-id p droplet-name)
        cert-name (name-from-domain domain)
        certk (ssl-certk cert-name)
        cert-id (ssl-cert-id p certk)
        lb-name (name-from-domain domain)
        lb-k (lbk lb-name)]
    {:data
     {lb-k
      (do/create-load-balancer
        token region lb-name cert-id did pleroma-port health-check-path)}
     :rollback [remove-load-balancer lb-k]}))

(defn lb-ip-k [lb-name]
  (-> (format "lbip-%s" lb-name) keyword))

(defn cleanup-k [_p _s _envk k]
  (infra/cleanup-data k))

(defn wait-for-lb-ip [p s _envk domain]
  (let [token (s :do-api-token)
        lb-name (name-from-domain domain)
        lb-k (lbk lb-name)
        lb-id (-> @p :data lb-k :load_balancer :id)
        lb-ip-k (lb-ip-k lb-name)]
    (infra/echo (format "Waiting for load balancer '%s' to be active" lb-name))
    (loop [current-status nil]
      (let [{{:keys [status]} :load_balancer :as new-result}
            (do/get-load-balancer-info token lb-id)]
        (when-not (= status current-status)
          (infra/echo 3 (format "Status changed from %s to %s"
                                current-status status)))
        (if (= "active" status)
          {:data {lb-ip-k new-result}
           :rollback [cleanup-k lb-ip-k]}
          (do (Thread/sleep 3000)
              (recur status)))))))

(defn- domain-a-records-key [domain]
  (->> domain (format "dom-a-rec-%s") keyword))

(defn delete-domain-records [p s _envk record-data-key]
  (let [token (s :do-api-token)
        {:keys [domain records]} (->> @p :data record-data-key)
        record-ids (map (comp :id :domain_record) records)]
    (infra/echo (format "Deleting domain records %s from %s"
                        (pr-str record-ids)
                        domain))
    (do/delete-domain-records token domain record-ids)
    (infra/cleanup-data record-data-key)))

(defn add-domain-a-records [p s _envk domain & subdomains]
  (let [token (s :do-api-token)
        lb-name (name-from-domain domain)
        lb-ip-k (lb-ip-k lb-name)
        ip (-> @p :data lb-ip-k :load_balancer :ip)
        data-key (domain-a-records-key domain)]

    (infra/echo (format "Adding A records to %s.%s -> %s" subdomains domain ip))
    {:data {data-key
            {:domain domain
             :records (do/add-domain-a-records token ip domain subdomains)}}
     :rollback [delete-domain-records data-key]}))

(defn upgrade-infra!
  [p s envk special-target]
  (let [env (name envk)
        friendly-env (str/replace env #"_" "-")
        droplet-name (str "pleroma-" friendly-env)
        container-name "pleroma"
        ssh-key-name "admin-ssh"
        dbc-name (format "pleroma-db-%s" friendly-env)
        db-name "pleroma"
        domain (s :root-domain)
        droplet-size "s-2vcpu-2gb"

        versions
        [1 [[upload-ssh-public-key ssh-key-name]]
         2 [[create-postgresql dbc-name]]
         3 [[create-droplet droplet-name droplet-size "docker-20-04"
             ssh-key-name]]
         4 [[wait-for-droplet droplet-name]]
         5 [[wait-for-db dbc-name]]
         6 [[add-postgresql-db dbc-name db-name]]
         7 [[configure-pleroma {:domain domain
                                :container-name container-name
                                :droplet-name droplet-name
                                :dbc-name dbc-name
                                :db-name db-name}]]
         8 [[build-pleroma droplet-name]]
         9 [[start-pleroma droplet-name]]
         10 [[wait-for-pleroma droplet-name]
             [add-pleroma-admin-user droplet-name container-name]]
         11 [[create-ssl-cert domain]]
         12 [[create-load-balancer droplet-name domain]]
         13 [[wait-for-lb-ip domain]]
         14 [[add-domain-a-records domain]]]]
    (infra/echo 1 (format "Domain %s" domain))

    (when (droplet-id p droplet-name)
      (infra/echo
        1
        (format "SSH into droplet using: '%s'"
                (str/join " " (sh/ssh-cmd (s :do-ssh-priv-key)
                                          (droplet-ip p s droplet-name))))))
    (cond
      (= special-target "dryrun")
      (infra/apply-infra! p s envk true false versions)

      (= special-target "debug")
      (infra/debug p s envk)

      (= special-target "restart")
      (infra/restart! p s envk)

      special-target
      (do (println "Invalid argument")
          (show-usage))

      :else
      (infra/apply-infra! p s envk false false versions))))

(defn -main
  "Get the infrastructure going"
  [& [env dest-version-or-cmd mode]]
  (if (not env)
    (show-usage)
    (let [to-file (-> (format "infra-%s.log" env)
                      io/file
                      infra/print-to-file-fn)
          envf (io/file "env" env)
          envk (keyword env)
          s (partial infra/secret-from-env env)]
      (.mkdirs envf)
      (infra/with-infra-tracking!
        (io/file envf "journal4")
        (fn [p]
          (binding [infra/*print* (fn [msg] (to-file msg) (println msg))]
            (try
              (let [{:keys [version]} @p
                    _ (if (re-matches #"^\d+$" (or dest-version-or-cmd ""))
                        (infra/rollback-infra!
                          p s envk
                          (Integer/parseInt dest-version-or-cmd)
                          (= "dryrun" mode)
                          (= "verbose" mode))
                        (upgrade-infra! p s envk dest-version-or-cmd))
                    {new-version :version} @p]
                (if-not (= version new-version)
                  (infra/echo
                    (format "Infra version of %s changed from %s to %s"
                            envk version new-version))))
              (catch Throwable t
                (let [stack-trace (with-out-str (st/print-cause-trace t))]
                  (infra/echo
                    (format "Infrastructure migration failed: '%s'%n%s"
                            (.getMessage t)
                            stack-trace))))))))))
  (shutdown-agents))
