(ns clj-infra.shell
  (:require [clj-infra.core :as infra]
            [clojure.java.shell :as sh]
            [clojure.string :as str]
            [clojure.java.io :as io])
  (:import (java.io File)))

(defn ssh-cmd [ssh-priv-key-file dip]
  ["ssh" "-o" "StrictHostKeyChecking=accept-new" "-i" ssh-priv-key-file
   (format "root@%s" dip)])

(defn- upload-file-cmd [ssh-priv-key-file dest-ip src dest]
  (str "cat "
       src
       " | "
       (str/join " " (ssh-cmd ssh-priv-key-file dest-ip))
       " -T 'cat > "
       dest
       "'"))

(defn exec [& args]
  (infra/echo (format "Executing: '%s'" (str/join " " args)))
  (let [{:keys [out err exit]} (apply sh/sh args)]
    (if (= 0 exit)
      out
      (throw (IllegalStateException. (format "Failed to execute '%s': %s/'%s'"
                                             args
                                             exit
                                             err))))))

(defn upload-file [ssh-priv-key-file dest-ip dest-filename contents]
  (infra/echo (format "Uploading '%s'" dest-filename))
  (let [temp-contents-base-file (File/createTempFile "infra" ".bin")
        temp-dir (doto (io/file "target" "infra") .mkdirs)
        temp-contents-file (io/file temp-dir (.getName temp-contents-base-file))

        upload-cmd (upload-file-cmd ssh-priv-key-file
                                    dest-ip
                                    (.getAbsolutePath temp-contents-file)
                                    dest-filename)

        temp-upload-script (File/createTempFile "upload" ".sh")]
    (spit temp-contents-file contents)
    (spit temp-upload-script upload-cmd)
    (exec "bash" (.getAbsolutePath temp-upload-script))
    (.delete temp-contents-base-file)
    (.delete temp-contents-file)
    (.delete temp-upload-script)))