(ns bunshin.core
  (:require [clojure.set :as cs]
            [ketamine.core :as ketama]
            [clj-time.core :as ct]
            [bunshin.datastores.redis :refer [redis-backend]]
            [bunshin.datastores.datastore :refer [BunshinDataStorage]]
            [bunshin.datastores.datastore :as bdd]))


(defn- gen-id-set-key
  [key]
  (format "bunshin-ids:%s" key))


(defn- gen-val-key
  [key id]
  (format "%s:%.0f" key (double id)))


(defn- gen-id
  []
  (.getMillis (ct/now)))


(defn- get-servers
  [ring id n]
  (take n (clojure.core/set (take (* n 2)
                                  (ketama/node-seq ring id)))))


(defn- get-fresh-id
  [server-with-id-xs]
  (first (first (first (sort-by (comp - first first)
                                (filter (comp seq first)
                                        server-with-id-xs))))))


(defn- fetch-id
  [{:keys [storage-backend]}
   server
   key]
  (when-let [id-str-xs (bdd/get-id-xs storage-backend
                                      server
                                      (gen-id-set-key key))]
    [(map (fn [i]
            (Double/parseDouble i))
          id-str-xs)
     server]))


(defn- fetch-id-xs
  [{:keys [^BunshinDataStorage storage-backend
           submit-to-threadpool-fn]
    :as ctx}
   servers
   key]
  (let [fetch-id-l (partial fetch-id ctx)
        results (map #(submit-to-threadpool-fn (fn []
                                              (fetch-id-l %
                                                          key)))
                     servers)]
    (doall (map deref
                results))))


(defn- set*
  [{:keys [^BunshinDataStorage storage-backend
           running-set-operations submit-to-threadpool-fn]}
   servers-with-id key val id
   & {:keys [ttl]}]
  (let [val-key (gen-val-key key id)]
    (when-not (@running-set-operations val-key)
      (swap! running-set-operations conj val-key)
      (doseq [[id-xs server] servers-with-id]
        (bdd/set storage-backend
                 server
                 val-key
                 val
                 (gen-id-set-key key)
                 id
                 ttl)
        (let [extra-ids (remove #(= (double id) %)
                                id-xs)]
          (when (seq extra-ids)
            (submit-to-threadpool-fn (fn []
                                       (bdd/prune-ids storage-backend
                                                      server
                                                      (gen-id-set-key key))
                                       (bdd/del storage-backend
                                                server
                                                (map (partial gen-val-key key)
                                                     extra-ids)))))))
      (swap! running-set-operations disj val-key))))


(defn gen-context
  "A function to generate context used by all API functions.

  servers-conf-list - A list of server configurations used by storage
  backend. This will be convereted into a ketama ring.

  storage-backend - By default gen-context will use redis-backend as
  storage backend but if you implement BunshinDataStorage you can pass
  any storage backend

  sumbit-to-threadpool-fn - All repair on reads and pruning old data on
  a key are submitted to this threadpool function. By default it's just
  a future. This is not advisible in production systems. You should
  provide your own implementation with this option.

  load-distribution-fn - This function is used by 'get' to decide which
  server to choose from a list of servers. List of servers is the
  argument to this function. Default behaviour just shuffles the list
  and picks the first entry"
  ([servers-conf-list]
     (gen-context servers-conf-list
                  redis-backend))
  ([servers-conf-list storage-backend]
     (gen-context servers-conf-list
                  storage-backend
                  (fn [thunk]
                    (future (thunk)))
                  (comp first shuffle)))
  ([servers-conf-list storage-backend
    submit-to-threadpool-fn load-distribution-fn]
     {:storage-backend storage-backend
      :submit-to-threadpool-fn submit-to-threadpool-fn
      :load-distribution-fn load-distribution-fn
      :running-set-operations (atom #{})
      :ring (ketama/make-ring servers-conf-list)}))


(defn set!
  "context - Use context generated by gen-context
   replication-factor - Number of copies you want of value
   ttl - time to live for the value
   id - A monotonically increasing number it will default to server timestamp

   Returns either :ok or :stale-write. :stale-write means the id
  provided is smaller than id already stored in datastore"
  [context key
   val & {:keys [replication-factor ttl id]
          :or {replication-factor 2
               ttl -1
               id (gen-id)}}]
  (let [servers (get-servers (:ring context) key replication-factor)
        servers-with-id (fetch-id-xs context servers key)
        fresh-id (get-fresh-id servers-with-id)]
    (if (or (nil? fresh-id)
            (<= fresh-id id))
      (do (set* context
                servers-with-id
                key
                val
                id
                :ttl ttl)
          :ok)
      :stale-write)))


(defn get-with-meta!
  "context - Use context generated by gen-context
   replication-factor - Number of copies you want of value
   ttl - time to live for the value

   Replication factor and ttl are needed for repair on read.

   Returns either nil or
   {:value   - Value for the key
    :servers - List of servers with this key
    :id      - Latest id of value}
   "
  [context key & {:keys [replication-factor ttl]
                  :or {replication-factor 2
                       ttl -1}}]
  (let [{:keys [ring load-distribution-fn storage-backend
                submit-to-threadpool-fn]} context
        servers (get-servers ring key replication-factor)]
    (let [servers-with-id (filter (comp seq first)
                                  (fetch-id-xs context servers key))]
      (when (seq servers-with-id)
        (let [fresh-id (get-fresh-id servers-with-id)]
          (when fresh-id
            (let [in-sync-servers (map second
                                       (filter #(= fresh-id (first (first %)))
                                               servers-with-id))
                  fresh-data (let [server (load-distribution-fn in-sync-servers)]
                               (bdd/get storage-backend
                                        server
                                        (gen-val-key key fresh-id)))]
              (submit-to-threadpool-fn
               (fn []
                 (let [out-of-sync-servers
                       (cs/difference (clojure.core/set servers)
                                      (clojure.core/set in-sync-servers))]
                   (set context
                        out-of-sync-servers
                        key
                        fresh-data
                        :id fresh-id
                        :ttl ttl
                        :replication-factor replication-factor))))
              {:value fresh-data
               :servers in-sync-servers
               :id fresh-id})))))))


(defn get!
  "context - Use context generated by gen-context
   replication-factor - Number of copies you want of value
   ttl - time to live for the value

   Returns either nil or value"
  [context key & {:keys [replication-factor ttl]
                  :or {replication-factor 2
                       ttl -1}}]
  (:value (get-with-meta! context key
                          :replication-factor replication-factor
                          :ttl ttl)))


(defn get-fast
  "context - Use context generated by gen-context

   Use id and servers returned from get-with-meta

   This will let you fetch data without in extra network hops. This
   function will return incosistent data if there are any writes between
   calling get-with-meta and get-fast. But this function lets you tune
   for performance at cost of incosistency"
  [context key id servers]
  (let [{:keys [load-distribution-fn storage-backend]} context
        val-key (gen-val-key key id)
        server (load-distribution-fn servers)]
    (bdd/get storage-backend
             server
             val-key)))


(defn del!
  "context - Use context generated by gen-context

  Delete key from all servers. Providing correct replication factor is
  important."
  [context key & {:keys [replication-factor id]
                  :or {replication-factor 2
                       id (gen-id)}}]
  (set! context key nil
        :replication-factor replication-factor
        :id id))


(comment
  (def ctx (gen-context [{:pool {}
                          :spec {:host "127.0.0.1"
                                 :port 6379}}]))

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"
  (get!  ctx "foo") ;; nil

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"
  ;;; Request 2 to 127.0.0.1:6379
  ;;; zadd "bunshinids:foo" 20 1
  ;;; set "foo:20" "hello world"
  (set! ctx "foo" "hello world" :id 20 :ttl 10) ;; :ok

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"
  (set! ctx "foo" "hello world new" :id 20) ;; :stale-write

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"

  ;;; Request 2 to 127.0.0.1:6379
  ;;; zadd "bunshinids:foo" 21 1
  ;;; set "foo:21" "hello worl new"
  ;;; zremrangebyrank "bunshin:foo" 1 -1
  ;;; del "foo:20"
  (set! ctx "foo" "hello world new" :id 21) ;; :ok

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"
  ;;; Request 2 to 127.0.0.1:6379
  ;;; get "foo:21"
  (get! ctx "foo") ;; "hello world new"

  (def ctx (gen-context [{:pool {}
                          :spec {:host "127.0.0.1"
                                 :port 6379}}
                         {:pool {}
                          :spec {:host "127.0.0.1"
                                 :port 6380}}
                         {:pool {}
                          :spec {:host "127.0.0.1"
                                 :port 6381}}
                         {:pool {}
                          :spec {:host "127.0.0.1"
                                 :port 6382}}]))


  ;; Assume that mapping for id foo is 127.0.0.1:6380 and 127.0.0.1:6381

  ;;; Request phase 1
  ;;; Requests to 127.0.0.1:6380, 127.0.0.1:6381
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"
  (get! ctx "foo") ;; nil

  ;;; Request phase 1
  ;;; Requests to 127.0.0.1:6380, 127.0.0.1:6381
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"

  ;;; Request phase 2
  ;;; Requests to 127.0.0.1:6380, 127.0.0.1:6381
  ;;; zadd "bunshinids:foo" 20 1
  ;;; set "foo:20" "hello world"
  (set! ctx "foo" "hello world" :id 20) ;; :ok

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"
  (set! ctx "foo" "hello world new" :id 20) ;; :stale-write

  ;;; Request 1 to 127.0.0.1:6379
  ;;; zrevrange "bunshinids:foo" 0 -1 "withscores"

  ;;; Request 2 to 127.0.0.1:6379
  ;;; zadd "bunshinids:foo" 21 1
  ;;; set "foo:21" "hello worl new"
  ;;; zremrangebyrank "bunshin:foo" 1 -1
  ;;; del "foo:20"
  (set! ctx "foo" "hello world new" :id 21) ;; :ok

  (def ctx (gen-context [{:pool {}
                          :spec {:host "127.0.0.1"
                                 :port 6379}}]))


  (get! ctx "foo") ;; served either from 6379
  )
