(ns compojure-api.core
  (:require [clojure.walk :as walk]
            [compojure.handler :as compojure]
            [ring.util.response :refer [response content-type]]
            [cheshire.core :as cheshire]
            [clojure.java.io :as io]
            [compojure.core :refer :all]))

(defn json? [{:keys [content-type] :as request}]
  (and
    content-type
    (not (empty? (re-find #"^application/(vnd.+)?json" content-type)))))

(defn wrap-json-body-and-params
  [handler]
  (fn [{:keys [character-encoding content-type body] :as request}]
    (println body)
    (handler
      (or
        (if (and body (json? request))
          (let [json (cheshire/parse-stream (io/reader body :encoding (or character-encoding "utf-8")))]
            (or
              (and (sequential? json)
                (-> request
                  (assoc request :body (vec json))))
              (and (map? json)
                (-> request
                  (assoc :body json)
                  (assoc :json-params json)
                  (update-in [:params] merge json))))))
        request))))

(defn wrap-json-response
  [handler]
  (fn [{:keys [body] :as request}]
    (let [response (handler request)]
      (if (coll? body)
        (-> response
          (content-type "application/json; charset=utf-8")
          (update-in [:body] cheshire/generate-string))
        response))))

(defn keywordize-request
  "keyworzies all ring-request keys recursively."
  [handler]
  (fn [request]
    (handler
      (walk/keywordize-keys request))))

(defmacro apiroutes [& body]
  `(-> (routes ~@body) api-middleware))

(defmacro defapi [name & body]
  `(defroutes ~name (apiroutes ~@body)))

(defn api-middleware
  [routes]
  (-> routes
    keywordize-request
    compojure/api
    wrap-json-response
    wrap-json-body-and-params))
