(ns compojure.api.resource
  (:require [compojure.api.routes :as routes]
            [compojure.api.meta :as meta]
            [schema.core :as s]))

(defn resource
  "Created a nested compojure-api Route from an enchanced ring-swagger operations map.
  Will do both request- and response-coercion based 

  Enchancements:

  1) special key `:handler` either under operations or at top-level. Value should be a
  ring-handler function, responsible for the actual request processing

  2) at top-level, one can add any ring-swagger operation definitions, which will be
  shared for all operations



  Top-level accepts any endpoint definitions, which
  are uses as defaults for the child operations.

  Ring-handler function is defined via :handler
  key either under operations or at top-level. Request

  Applies request and response coercion based on
  those definitions."
  [info]
  (let [methods #{:get :head :patch :delete :options :post :put}
        parameter-mapping {:query [:query-params :string true]
                           :body [:body-params :body false]
                           :formData [:form-params :string true]
                           :header [:header-params :string true]
                           :path [:path-params :string true]}
        root-info (reduce dissoc info methods)
        child-info (select-keys info methods)
        childs (map (fn [[method info]] (routes/create "/" method info nil nil)) child-info)
        coerce-request (fn [request ks]
                         (reduce-kv
                           (fn [request k [v type open?]]
                             (if-let [schema (get-in info (concat ks [:parameters k]))]
                               (let [schema (if open? (assoc schema s/Keyword s/Any) schema)]
                                 (update request v merge (meta/coerce! schema v type request)))
                               request))
                           request
                           parameter-mapping))
        coerce-response (fn [response request ks]
                          (meta/coerce-response! request response (get-in info (concat ks [:responses]))))
        resolve-handler (fn [request-method]
                          (or
                            (get-in info [:handler])
                            (get-in info [request-method :handler])
                            (throw (ex-info (str "No handler defined for" request-method) info))))
        handler (fn [{:keys [request-method] :as request}]
                  (-> ((resolve-handler request-method)
                        (-> request
                            (coerce-request [])
                            (coerce-request [request-method])))
                      (coerce-response request [request-method])
                      (coerce-response request [])))]
    (routes/create nil nil root-info childs handler)))
