(ns clj-base62.core
  (:require [clojure.string :as str])
  (:import java.nio.ByteBuffer))

(def +charset+
  "The character set used by Base62 encoding."
  "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz")

;; We handle leading zeros the same way as base-x npm package:
;; Encoding: each leading zero byte results in a 0 char.
;; Decoding: each leading 0 char results in a zero byte.

(defn encode
  "Encode byte array `b` into a base62 string."
  [^"[B" b]
  (if (empty? b)
    ""
    (let [s (StringBuilder.)
          zero-count (count (take-while zero? b))]
      ;; BigInteger's signum must be 1 so that b is processed unsigned
      (loop [i (BigInteger. 1 b)]
        (when-not (zero? i)
          (.append s (nth +charset+ (mod i 62)))
          (recur (quot i 62))))
      (str (str/join (repeat zero-count "0")) (.reverse s)))))

(defn- char-index [c]
  (if-let [index (str/index-of +charset+ c)]
    index
    (throw (ex-info (str "Character " (pr-str c) " is not part of Base62 character set.")
                    {:type ::illegal-character
                     :character c}))))

(defn decode
  "Decode base62 string `s` into a byte array."
  [s]
  (if (empty? s)
    (byte-array 0)
    (let [[zeros rest] (split-with #(= % \0) s)
          rest-bytes (-> (reduce (fn [i c] (+ (* i 62) (char-index c)))
                                 (bigint 0)
                                 rest)
                         (biginteger)
                         (.toByteArray))
          zero-count (count zeros)
          ;; In case the most significant byte would be negative, there's a zero
          ;; sign byte. We'll drop that as we always process the number as
          ;; unsigned.
          rest-offset (if (= 0 (aget rest-bytes 0)) 1 0)
          rest-count (- (count rest-bytes) rest-offset)]
      (-> (ByteBuffer/allocate (+ zero-count rest-count))
          (.position zero-count)
          (.put rest-bytes rest-offset rest-count)
          (.array)))))
