;;  
;;   Copyright (c) Steve Lindsay, 2012. All rights reserved.
;;
;;   The use and distribution terms for this software are covered by the
;;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;;   which can be found in the file epl-v10.html at the root of this 
;;   distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;
;;   You must not remove this notice, or any other, from this software.
;;

(ns minnow.nrepl
  (:gen-class)
  (:use
    [clojure.tools.logging :only [info error debug]])
  (:require 
    [clojure.tools.nrepl.transport]
    [clojure.pprint :as pp]
    [clojure.tools.nrepl.server :as server]
    [clojure.tools.nrepl :as nrepl]
    [clojure.tools.nrepl.ack :as ack]
    [clj-logging-config.log4j :as log-config])
  (:import
    clojure.tools.nrepl.transport.Transport
    [org.apache.log4j EnhancedPatternLayout DailyRollingFileAppender RollingFileAppender]))

(def log-file (format "%s/.minnow/nrepl.log" (System/getProperty "user.home")))

(log-config/set-logger! :out (org.apache.log4j.DailyRollingFileAppender.
                               (org.apache.log4j.EnhancedPatternLayout. "%d %-5p %c - %m%n")
                               log-file
                               ".yyyy-MM-dd"))

(defn pr-values
  [h]
  (fn [{:keys [op ^Transport transport] :as msg}]
    (if (not= op "eval")
      (h msg)
      (h (assoc msg :transport (reify Transport
                                 (recv [this] (.recv transport))
                                 (recv [this timeout] (.recv transport timeout))
                                 (send [this resp]
                                   (.send transport
                                     (if-let [[_ v] (find resp :value)]
                                       (if clojure.pprint/*print-pretty*
                                         (assoc resp :value (with-out-str (pp/pprint v)))
                                         (assoc resp :value (with-out-str (pr v))))
                                       resp)))))))))

(defn logger
  [h]
  (fn [msg]    
    (debug msg)
    (h msg)))

(defn handler
  []
  (-> (constantly false)    
    clojure.tools.nrepl.middleware.interruptible-eval/interruptible-eval
    logger    
    pr-values   
    clojure.tools.nrepl.middleware.session/add-stdin
    clojure.tools.nrepl.middleware.session/session))

(defn run
  [ack-port]
  (let [server (server/start-server :port 0 :ack-port ack-port :handler (handler))]
    (info "nREPL server started, listening on port: " (.getLocalPort (:ss server)))))

(defn -main 
  [ack-port]
  (run (Integer. ack-port)))

