# multi-atom

An abstraction for performing swap! to atomic cells in some larger database or store. Clojure's time model is great, and I feel the most widely used time abstraction in clojure, the atom, can be implemented effectively against (remote/distributed) data stores. 

But most databases are not atoms. Though they sometimes have things within them that can (values, records, rows). Therefore there exists a mapping from some key or identifier to a thing that can behave like an atom. 

A multi-atom can be described as a map of keys to cells, each cell is itself an atom - it can be changed atomically from one value to another via function application. It is always possible to derive an atom from a multi-atom.

API docs can be found [here](http://danstone.github.io/multi-atom).

## Currently Supported atoms
- local (just an `clojure.lang.Atom` with a map in it)
- Amazon [DynamoDB](https://aws.amazon.com/dynamodb/) via [faraday](https://github.com/ptaoussanis/faraday)

New atoms can be created by extending the protocols `IMultiAtom` and `IMultiDeref` in `multi-atom.core`.

## Installation

Drop into your project.clj dependencies

`[mixradio/multi-atom "0.1.0"]`

For DynamoDB include `[com.taoensso/faraday "1.7.1"]`

## Usage

With 2 functions in `multi-atom.core` just `swap-at!` and `deref-at`.

As an example I'll show using the dynamo db multi-atom.

```clojure
(require '[multi-atom.core refer [deref-at swap-at! atom-view]
         '[multi-atom.aws.dynamo :as dynamo])
;;; A table client can be created with dynamo/table-client, we will cover that in the dynamo section
(def multi-atom (dynamo/multi-atom my-table-client))

(deref-at multi-atom :foo) => nil ;; there is nothing in the cell :foo right now
(deref-at multi-atom [1 :foo []]) => nil ;; the keys used are arbitrary as long as they are edn**

;;lets seed a cell
(swap-at! multi-atom :foo (constantly 0)) => 0 ;; Just like swap!
;; I can apply a fn & args
(swap-at! multi-atom "fred" assoc :hello :world) => {:hello :world} ;; I can store any edn data I like
;; I can get the value back out again
(deref-at multi-atom "fred") => {:hello :world} ;; (for dynamo, a consistent read is used)
```

## But I want a clojure.lang.IAtom
You can use `atom-view`.
```clojure
(def atom (atom-view multi-atom :foo))
(instance? clojure.lang.IAtom atom) => true
@atom => 0
(deref atom) => 0
(swap! atom inc) => 1
@atom => 1
```

## Amazon DynamoDB

Make sure you have `[com.taoensso/faraday "1.7.1"]` on in your dependencies to use the dynamo backend.

A `table-client` is required to create an atom, it is a record made up of:
 - The [faraday](https://github.com/ptaoussanis/faraday) client opts map (dynamo endpoint, thread pool settings etc)
 - The table name, this is a keyword
 - A key name, this is the primary hash key column to use. 
 
``` clojure
(def table-client (dynamo/table-client faraday-client-opts :table-name :key-name))
```

The best way to create a table suitable for the atom is via `create-table!` or `ensure-table!`, this will ensure the primary key is encoded correctly etc.
```clojure
(ensure-table! table-client) ;; will create the table with some default provision settings
```
The hash key schema on the table must be a string (as all keys will be encoded as strings), the key-name will also be encoded via `write-key`. You can use `write-key` on a value to see what the representation of it will be if it is used as a key.

All edn data structures are supported by the multi-atom**. However they are encoded using dynamo columns and data types where possible to enable query and indexing on multi-atom tables. Checkout your table with the amazon console to see what is going on.

**  If you use datastructures like sets or maps that rely on hashes for ordering as keys, and the implementation of the printer/reader/hashing function changes (with a new clojure version) - you may not be able to `deref-at` your data, as the backend will need to print those datastructures to use them as keys.

*** range keys are not currently supported by the provided Dynamo back end.

Various useful functions can be found in `multi-atom.aws.dynamo` to do reads and writes a-la-carte providing the same encoding scheme as the atom, so for example you can do batch gets with `find-items`.

## Contributing

Easy (and welcome) ones would be Zookeeper, Relational databases, anything with CAS and consistent reads.

- Support all edn data as best as possible, a superset is good too (e.g DynamoDB multi-atoms support byte-arrays)
- Don't just wrap everything in a string or byte array if it can be helped, allow the native features of the 
 data store to be used (e.g indexes, query) outside of the atom context.
- Hold to the guarantees expected of a user of an atom (consistent read, atomic swap)
- Any additional dependencies for your ns should be :dev only please.

## License

Copyright © 2015 MixRadio

[multi-atom is released under the 3-clause license ("New BSD License" or "Modified BSD License").](http://github.com/mixradio/multi-atom/blob/master/LICENSE)
