(ns multi-atom.core
  "The core abstraction")

(defprotocol IMultiDeref
  (-deref-at [this key not-found]))

(defn deref-at
  "Derefs the value in the cell given by `key`"
  ([ref key]
   (deref-at ref key nil))
  ([ref key not-found]
   (-deref-at ref key not-found)))

(defprotocol IMultiAtom
  (-swap-at! [this key f]))

(defn swap-at!
  "Performs a `swap!` at the given `key`
   For use on members of `IMultiAtom`."
  [multi-atom key f & args]
  (-swap-at! multi-atom key #(apply f % args)))

;;all atoms containing maps are IMultiAtom's
(extend-type clojure.lang.IAtom
  IMultiAtom
  (-swap-at! [this key f]
    (get (swap! this update key f) key)))

;;all atoms containing maps are IDerefCell
(extend-type clojure.lang.IDeref
  IMultiDeref
  (-deref-at [this key not-found]
    (get (deref this) key not-found)))

(defn atom-view
  "Returns a view on the given multi-atom at `key`
   that implements `clojure.lang.IDeref` and `clojure.lang.IAtom`."
  [multi-atom key]
  (reify clojure.lang.IDeref
    (deref [this] (deref-at multi-atom key))
    clojure.lang.IAtom
    (swap [this f]
      (-swap-at! multi-atom key f))
    (swap [this f x]
      (-swap-at! multi-atom key #(f % x)))
    (swap [this f x y]
      (-swap-at! multi-atom key #(f % x y)))
    (swap [this f x y args]
      (-swap-at! multi-atom key #(apply f % x y args)))
    (compareAndSet [this old new]
      (swap! this #(if (= old %) new %)))
    (reset [this v]
      (swap! this (constantly v)))))

(defn namespaced-multi-atom
  "Prefixes all cell accesses via the multi-atom with `ns`"
  [multi-atom ns]
  (if (some? ns)
    (reify IMultiAtom
      (-swap-at! [this key f]
        (-swap-at! multi-atom [ns key] f))
      IMultiDeref
      (-deref-at [this key not-found]
        (-deref-at multi-atom [ns key] not-found)))
    multi-atom))