(ns modelizer.core
  (:refer-clojure :exclude [get-validator])
  (:require
    #?(:cljs [cljs.core :refer [Atom]])
    [modelizer.helpers :as helpers]
    [modelizer.validator :as mv]
    [modelizer.generator :as mg]
    [modelizer.protocols.schema :as proto.schema]
    [modelizer.protocols.registry :as proto.registry])
  #?(:clj
     (:import
       (java.io Writer)
       (clojure.lang Atom))))

#?(:clj (set! *warn-on-reflection* true))



;;;;
;; Schema
;;;;

(def ^{:added "0.0.2", :private true}
  +schema-type+
  "A schema type."
  :modelizer/schema)


(def ^{:added "0.0.2", :dynamic true}
  *default-schema-version*
  "A default schema version."
  :schema.version/not-specified)


(def ^{:added "0.0.2", :dynamic true}
  *default-schema-status*
  "A default schema status."
  :schema.status/not-specified)


(def ^{:added "0.0.2", :dynamic true}
  *default-namespace-aliases*
  {'cljs.core      nil
   'clojure.core   nil
   'clojure.string 'string})


(def ^{:added "0.0.2", :dynamic true}
  *default-sample-generation-maximum-tries*
  "A default sample generation maximum tries."
  100)


(def ^{:added "0.0.4"}
  default-generator-options
  "A default generator options."
  {:max-tries *default-sample-generation-maximum-tries*})


;; NOTE: Add default strategy and comparator for getting schema version
;; NOTE: Make the creation of validators, explainers and generators in a single pass?

(declare schema?
  into-schema get-schema
  into-validator get-validator
  into-explainer get-explainer
  into-generator get-generator)



;;;;
;; Helper functions
;;;;

(def ^{:added "0.0.4"}
  special-forms
  "A special form types."
  #{:and :or :not
    :< :<= := :> :>= :not=
    :enum
    :coll-of
    :tuple :map-of :map})


(defn special-form?
  "Returns `true` if the given value is special form. Otherwise `false`."
  {:added "0.0.2"}
  [x]
  (and
    (sequential? x)
    (contains? special-forms (first x))))


(defn resolve-ns-alias
  "Returns a default namespace alias."
  {:added "0.0.2"}
  [ns]
  (let [alias (get *default-namespace-aliases* ns)]
    (when (contains? *default-namespace-aliases* alias)
      alias)))


(defn get-schema-alias
  "Returns a schema alias specified in the metadata."
  {:added "0.0.6"}
  [key meta]
  (get-in meta [:require key]))


(defn keyword->schema-version
  "Returns a schema alias specified in the metadata as a keyword."
  {:added "0.0.6"}
  [key meta]
  (let [name (get-schema-alias key meta)]
    [name *default-schema-version*]))


(defn vector->schema-version
  "Returns a schema alias specified in the metadata as a vector (schema name with version)."
  {:added "0.0.6"}
  [key meta]
  (let [[name version] (get-schema-alias key meta)]
    [name (or version *default-schema-version*)]))


(defn nil->schema-version
  "Returns a schema name with the default version when schema alias isn't specified in the metadata."
  {:added "0.0.6"}
  [key _]
  [key *default-schema-version*])


(defn resolve-schema-version
  "Returns a schema name and version."
  {:added "0.0.6"}
  [key meta]
  (let [x (get-schema-alias key meta)]
    (cond
      (nil? x) (nil->schema-version key meta)
      (keyword? x) (keyword->schema-version key meta)
      (vector? x) (vector->schema-version key meta)
      :else
      (let [msg (helpers/format "Unable to resolve a schema alias - the given key `%s` isn't found in the metadata" key)]
        (throw (ex-info msg {:key key, :meta meta}))))))



;;;;
;; Schema generator builders
;;;;

(defn fn->generator
  "Creates a schema generator from the given function."
  {:arglists '([f meta])
   :added    "0.0.3"}
  [f _]
  (mg/make-generator f {}))


(defn keyword->generator
  "Creates a schema generator from the given keyword."
  {:added "0.0.3"}
  [key meta]
  (let [[name version] (resolve-schema-version key meta)]
    (if-some [schema (get-schema name version)]
      (get-generator schema)
      (let [msg (helpers/format "Can't find schema `%s` in registry by the given alias `[%s %s]`" key name version)]
        (throw (ex-info msg {:key key, :alias [name version], :meta meta}))))))


(defmulti special-form->generator
  "Creates a schema generator from the special form."
  {:arglists '([form meta])
   :added    "0.0.3"}
  (fn [form _]
    (first form)))


(defmethod special-form->generator :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema generator from the special form - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod special-form->generator :and
  [form meta]
  (let [valid? (into-validator form meta)]
    (as-> form $
      (rest $)
      (map #(into-generator % meta) $)
      (mg/gen-one-of $)
      (mg/gen-such-that valid? $ default-generator-options)
      (into-generator $ meta))))


(defmethod special-form->generator :or
  [form meta]
  (let [valid? (into-validator form meta)]
    (as-> form $
      (rest $)
      (map #(into-generator % meta) $)
      (mg/gen-one-of $)
      (mg/gen-such-that valid? $ default-generator-options)
      (into-generator $ meta))))


(defmethod special-form->generator :not
  [form meta]
  (let [valid?    (into-validator form meta)
        generator (mg/gen-such-that valid? mg/gen-any? default-generator-options)]
    (into-generator generator meta)))


(defmethod special-form->generator :<
  [form meta]
  (let [valid?    (into-validator form meta)
        max       (dec (second form))
        generator (as-> (mg/make-generator max {:max max}) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :<=
  [form meta]
  (let [valid?    (into-validator form meta)
        max       (second form)
        generator (as-> (mg/make-generator max {:max max}) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :=
  [form meta]
  (-> form second mg/gen-return (into-generator meta)))


(defmethod special-form->generator :>
  [form meta]
  (let [valid?    (into-validator form meta)
        min       (inc (second form))
        generator (as-> (mg/make-generator min {:min min}) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :>=
  [form meta]
  (let [valid?    (into-validator form meta)
        min       (second form)
        generator (as-> (mg/make-generator min {:min min}) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :not=
  [form meta]
  (let [valid?    (into-validator form meta)
        value     (second form)
        generator (as-> (mg/make-generator value {}) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :enum
  [form meta]
  (into-generator (mg/gen-elements (rest form)) meta))


(defmethod special-form->generator :coll-of
  [form meta]
  (let [[_ value & {:as opts}] form
        kind      (get opts :kind :coll)
        size      (:count opts)
        min       (or size (:min opts))
        max       (or size (:max opts))
        unique?   (get opts :distinct false)
        valid?    (into-validator form meta)
        gen-value (into-generator value meta)
        gen-opts  (assoc default-generator-options
                    :gen gen-value :kind kind :min min :max max :distinct unique?)
        generator (as-> (mg/make-generator kind gen-opts) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :tuple
  [form meta]
  (let [valid?    (into-validator form meta)
        gen-value (->> form rest (map #(into-generator % meta)) (apply mg/gen-tuple))
        generator (mg/gen-such-that valid? gen-value default-generator-options)]
    (into-generator generator meta)))


(defmethod special-form->generator :map-of
  [form meta]
  (let [[_ key value & {:as opts}] form
        size      (:count opts)
        min       (or size (:min opts))
        max       (or size (:max opts))
        gen-key   (into-generator key meta)
        gen-value (into-generator value meta)
        gen-opts  (assoc default-generator-options
                    :gen-key gen-key :gen-value gen-value :min min :max max)
        valid?    (into-validator form meta)
        generator (as-> (mg/gen-map-of gen-opts) $
                    (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defmethod special-form->generator :map
  [form meta]
  (let [entries       (reduce
                        (fn [acc v]
                          (if (keyword? v)
                            (conj acc [v v])
                            (conj acc v)))
                        [] (rest form))
        optional-keys (-> meta (get-in [:rules :optional]) set)
        generators    (reduce
                        (fn [acc [key value]]
                          (let [generator (mg/gen-fmap (fn [v] [key v]) (into-generator value meta))
                                generator (if-not (contains? optional-keys key)
                                            generator
                                            (mg/gen-one-of [(mg/gen-return nil) generator]))]
                            (conj acc generator)))
                        [] entries)
        valid?        (into-validator form meta)
        f             (partial into {})
        generator     (as-> generators $
                        (apply mg/gen-tuple $)
                        (mg/gen-fmap f $)
                        (mg/gen-such-that valid? $ default-generator-options))]
    (into-generator generator meta)))


(defn into-generator
  "Creates a schema generator."
  {:added "0.0.3"}
  [form meta]
  (cond
    (mg/generator? form) form
    (special-form? form) (special-form->generator form meta)
    (keyword? form) (keyword->generator form meta)
    (var? form) (fn->generator @form meta)
    (fn? form) (fn->generator form meta)
    (schema? form) (get-generator form)
    :else
    (let [type (type form)
          msg  (helpers/format "Unable to create a schema generator - the given type `%s` isn't implemented" type)]
      (throw (ex-info msg {:type type, :form form, :meta meta})))))



;;;;
;; Schema validator builders
;;;;

(defn fn->validator
  "Creates a schema validator from the given function."
  {:arglists '([f meta])
   :added    "0.0.2"}
  [f _]
  (fn validator
    ([data]
     (try (f data) (catch #?(:clj Exception :cljs js/Error) _ false)))
    ([data meta]
     (try (f data meta) (catch #?(:clj Exception :cljs js/Error) _ false)))))


(defn keyword->validator
  "Creates a schema validator from the given keyword."
  {:added "0.0.2"}
  [key meta]
  (let [[name version] (resolve-schema-version key meta)]
    (if-some [schema (get-schema name version)]
      (get-validator schema)
      (let [msg (helpers/format "Can't find schema `%s` in registry by the given alias `[%s %s]`" key name version)]
        (throw (ex-info msg {:key key, :alias [name version], :meta meta}))))))


(defmulti special-form->validator
  "Creates a schema validator from the special form."
  {:arglists '([form meta])
   :added    "0.0.2"}
  (fn [form _]
    (first form)))


(defmethod special-form->validator :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema validator from the special form - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod special-form->validator :and
  [form meta]
  (let [valid? (->> form rest (map #(into-validator % meta)) (apply every-pred))]
    (fn->validator valid? meta)))


(defmethod special-form->validator :or
  [form meta]
  (let [valid? (->> form rest (map #(into-validator % meta)) (apply some-fn))]
    (fn->validator valid? meta)))


(defmethod special-form->validator :not
  [form meta]
  (let [valid? (complement (into-validator (cons :or (rest form)) meta))]
    (fn->validator valid? meta)))


(defmethod special-form->validator :<
  [form meta]
  (let [max    (second form)
        valid? #(< % max)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :<=
  [form meta]
  (let [max    (second form)
        valid? #(<= % max)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :=
  [form meta]
  (let [value  (second form)
        valid? #(= % value)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :>
  [form meta]
  (let [min    (second form)
        valid? #(> % min)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :>=
  [form meta]
  (let [min    (second form)
        valid? #(>= % min)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :not=
  [form meta]
  (let [value  (second form)
        valid? #(not= % value)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :enum
  [form meta]
  (let [coll   (set (rest form))
        valid? (partial contains? coll)]
    (fn->validator valid? meta)))


(defmethod special-form->validator :coll-of
  [form meta]
  (let [[_ value & {:as opts}] form
        kind              (get opts :kind :coll)
        size              (:count opts)
        min               (or size (:min opts))
        max               (or size (:max opts))
        unique?           (get opts :distinct false)
        valid-type?       (case kind
                            :list list?
                            :vector vector?
                            :set set?
                            :sequential sequential?
                            :coll coll?
                            (constantly false))
        valid-size?       (cond
                            (not (or min max)) (constantly true)
                            (and min max) #(<= min (count %) max)
                            min #(<= min (count %))
                            max #(<= (count %) max)
                            :else (constantly false))
        valid-value?      (into-validator value meta)
        valid-values?     (fn [coll]
                            (boolean
                              #?(:clj
                                 (let [it (.iterator ^Iterable coll)]
                                   (boolean
                                     (loop []
                                       (if (.hasNext it)
                                         (and (valid-value? (.next it)) (recur))
                                         true))))
                                 :cljs
                                 (reduce
                                   (fn [_ x]
                                     (or (valid-value? x)
                                       (reduced false)))
                                   true coll))))
        valid-uniqueness? (if-not unique?
                            (constantly true)
                            (fn [coll]
                              (cond
                                (empty? coll) true
                                (set? coll) true
                                :else (apply distinct? coll))))
        valid?            (fn [coll]
                            (and (valid-type? coll)
                              (valid-size? coll)
                              (valid-values? coll)
                              (valid-uniqueness? coll)))]
    (fn->validator valid? meta)))


(defmethod special-form->validator :tuple
  [form meta]
  (let [validators (->> form rest (map #(into-validator % meta)) (map-indexed vector) (into (array-map)))
        size       (count validators)
        valid?     (fn [coll]
                     (and
                       (vector? coll)
                       (= size (count coll))
                       (reduce-kv
                         (fn [_ idx validator]
                           (or (validator (nth coll idx))
                             (reduced false)))
                         true validators)))]
    (fn->validator valid? meta)))


(defmethod special-form->validator :map-of
  [form meta]
  (let [[_ key value & {:as opts}] form
        size         (:count opts)
        min          (or size (:min opts))
        max          (or size (:max opts))
        valid-key?   (into-validator key meta)
        valid-value? (into-validator value meta)
        valid-size?  (cond
                       (not (or min max)) (constantly true)
                       (and min max) #(<= min (count %) max)
                       min #(<= min (count %))
                       max #(<= (count %) max)
                       :else (constantly false))
        valid-map?   (fn [m]
                       (reduce-kv
                         (fn [_ k v]
                           (or (and (valid-key? k) (valid-value? v))
                             (reduced false)))
                         true m))
        valid?       (fn [m]
                       (and (map? m)
                         (valid-size? m)
                         (valid-map? m)))]
    (fn->validator valid? meta)))


(defmethod special-form->validator :map
  [form meta]
  (let [entries          (reduce
                           (fn [acc v]
                             (if (keyword? v)
                               (conj acc [v v])
                               (conj acc v)))
                           [] (rest form))
        defined-keys     (->> entries (map first) set)
        optional-keys    (-> meta (get-in [:rules :optional]) set)
        closed?          (get-in meta [:rules :closed] false)
        value-validators (mapv
                           (fn [[k v]]
                             (let [valid?    (into-validator v meta)
                                   optional? (contains? optional-keys k)]
                               (fn [m]
                                 (if-let [entry (find m k)]
                                   (valid? (val entry))
                                   optional?))))
                           entries)
        validators       (if-not closed?
                           value-validators
                           (conj value-validators
                             (fn [m]
                               (reduce
                                 (fn [_ k]
                                   (or (contains? defined-keys k)
                                     (reduced false)))
                                 true (keys m)))))
        valid-map?       (fn [m]
                           (boolean
                             #?(:clj
                                (let [it (.iterator ^Iterable validators)]
                                  (boolean
                                    (loop []
                                      (if (.hasNext it)
                                        (and ((.next it) m) (recur))
                                        true))))
                                :cljs
                                (reduce
                                  (fn [_ validator]
                                    (or (validator m)
                                      (reduced false)))
                                  true validators))))
        valid?           (fn [m]
                           (and (map? m)
                             (valid-map? m)))]
    (fn->validator valid? meta)))


(defn into-validator
  "Creates a schema validator."
  {:added "0.0.2"}
  [form meta]
  (cond
    (keyword? form) (keyword->validator form meta)
    (var? form) (fn->validator @form meta)
    (fn? form) (fn->validator form meta)
    (special-form? form) (special-form->validator form meta)
    (schema? form) (get-validator form)
    :else
    (let [type (type form)
          msg  (helpers/format "Unable to create a schema validator - the given type `%s` isn't implemented" type)]
      (throw (ex-info msg {:type type, :form form, :meta meta})))))



;;;;
;; Schema explainer builders
;;;;

;; TBD



;;;;
;; Schema form builders
;;;;

(defn into-form
  "Creates a schema form."
  {:added "0.0.2"}
  [form meta]
  (cond
    (special-form? form) form
    (keyword? form) form
    (var? form) form
    (fn? form) form
    :else
    (let [type (type form)
          msg  (helpers/format "Unable to create a schema form - the given form type `%s` isn't implemented" type)]
      (throw (ex-info msg {:type type, :form form, :meta meta})))))



;;;;
;; Schema builders
;;;;

(defn map->schema
  "Creates a schema from the given map."
  {:added "0.0.2"}
  [map]
  (let [name      (:name map)
        meta      (get map :meta {})
        form      (into-form (:form map) meta)
        validator (delay (into-validator form meta))
        explainer (delay (fn []))
        generator (delay (into-generator form meta))]
    {:name      name
     :meta      meta
     :form      form
     :validator validator
     :explainer explainer
     :generator generator}))


(defn var->schema
  "Creates a schema from the given var (clojure.lang.Var)."
  {:added "0.0.2"}
  ([var]
   (let [var-meta (meta var)
         var-name (str (:name var-meta))
         var-ns   (some-> var-meta :ns ns-name symbol)
         alias    (resolve-ns-alias var-ns)
         name     (keyword (some-> alias str) var-name)]
     (var->schema name var)))

  ([name var]
   (let [var-meta (meta var)
         var-ns   (some-> var-meta :ns ns-name symbol)
         added    (:added var-meta)
         doc      (:doc var-meta)
         meta     (cond-> {}
                    (some? doc) (assoc :doc doc)
                    (some? var-ns) (assoc :ns var-ns)
                    (some? added) (assoc :added added))]
     (map->schema {:name name, :meta meta, :form var}))))


(defn into-schema
  "Creates a schema."
  {:added "0.0.2"}
  [x]
  (cond
    (map? x) (map->schema x)
    (var? x) (var->schema x)
    (schema? x) x
    :else
    (let [type (type x)
          msg  (helpers/format "Unable to create a schema - the given type `%s` isn't implemented" type)]
      (throw (ex-info msg {:type type, :schema x})))))


(defn schema?
  "Returns `true` if the given schema is implements `modelizer.protocols.schema/Schema` protocol.
  Otherwise `false`."
  {:added "0.0.2"}
  [x]
  (satisfies? proto.schema/Schema x))


(defn schema
  "Creates a schema."
  {:arglists '([schema])
   :added    "0.0.2"}
  [?schema]
  (if (schema? ?schema)
    ?schema
    (let [s         (into-schema ?schema)
          name      (:name s)
          meta      (:meta s)
          form      (:form s)
          version   (or (:version meta) *default-schema-version*)
          status    (or (:status meta) *default-schema-status*)
          doc       (:doc meta)
          validator (:validator s)
          explainer (:explainer s)
          generator (:generator s)]
      ^{:type +schema-type+}
      (reify proto.schema/Schema
        (-name [_] name)
        (-version [_] version)
        (-status [_] status)
        (-doc [_] doc)
        (-form [_] form)
        (-meta [_] meta)
        (-unwrap [_] s)
        (-validator [_] @validator)
        (-validate [_ data] (@validator data))
        (-validate [_ data context] (@validator data context))
        (-explainer [_] @explainer)
        (-explain [_ data] (@explainer data))
        (-explain [_ data context] (@explainer data context))
        (-generator [_] @generator)
        (-generate [_] (mg/generate @generator))
        (-generate [_ opts] (mg/generate @generator opts))
        (-sample [_] (mg/sample @generator))
        (-sample [_ n] (mg/sample @generator n))
        (-sample [_ n opts] (mg/sample @generator n opts))

        #?@(:cljs
            [IPrintWithWriter
             (-pr-writer [x w opts]
               (-write w (helpers/keyword->prefix +schema-type+))
               (-pr-writer (proto.schema/-unwrap x) w opts))])))))


(defn get-name
  "Returns a schema name."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-name schema))


(defn get-version
  "Returns a schema version."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-version schema))


(defn get-status
  "Returns a schema status."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-status schema))


(defn get-doc
  "Returns a schema doc."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-doc schema))


(defn get-form
  "Returns a schema form."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-form schema))


(defn get-meta
  "Returns a schema metadata."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-meta schema))


(defn unwrap
  "Returns a schema data."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-unwrap schema))


(defn get-validator
  "Returns a schema validator."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-validator schema))


(defn validate
  "Returns `true` if the given `data` is valid for the `schema`. Otherwise `false`."
  {:added "0.0.2"}
  ([schema data]
   (proto.schema/-validate schema data))

  ([schema data context]
   (proto.schema/-validate schema data context)))


(defn get-explainer
  "Returns a schema explainer."
  {:added "0.0.10"}
  [schema]
  (proto.schema/-explainer schema))


(defn explain
  "Explains errors in the given `data`. Otherwise `nil`."
  {:added "0.0.10"}
  ([schema data]
   (proto.schema/-explain schema data))

  ([schema data context]
   (proto.schema/-explain schema data context)))


(defn get-generator
  "Returns a schema generator."
  {:added "0.0.3"}
  [schema]
  (proto.schema/-generator schema))


(defn generate
  "Generates a single sample value from the schema generator."
  {:added "0.0.3"}
  ([schema]
   (proto.schema/-generate schema))

  ([schema opts]
   (proto.schema/-generate schema opts)))


(defn sample
  "Generates samples from the schema generator."
  {:added "0.0.3"}
  ([schema]
   (proto.schema/-sample schema))

  ([schema n]
   (proto.schema/-sample schema n))

  ([schema n opts]
   (proto.schema/-sample schema n opts)))



;;;;
;; Registry
;;;;

(def ^{:added "0.0.2", :private true}
  +registry-type+
  "A registry type."
  :modelizer/registry)


(defn registry?
  "Returns `true` if the given schema is implements `modelizer.protocols.registry/Registry` protocol.
  Otherwise `false`."
  {:added "0.0.2"}
  [x]
  (satisfies? proto.registry/Registry x))


(defn registry
  "Creates a registry."
  {:added "0.0.2"}
  ([]
   (registry (atom {})))

  ([^Atom *registry]
   ^{:type +registry-type+}
   (reify proto.registry/Registry
     (-schemas [_] @*registry)
     (-schemas [_ name] (get @*registry name))
     (-schema [_ name] (get-in @*registry [name *default-schema-version*]))
     (-schema [_ name version] (get-in @*registry [name version]))
     (-register [_ ?schema]
       (let [s    (schema ?schema)
             path [(get-name s) (get-version s)]]
         (swap! *registry assoc-in path s)
         s))

     #?@(:cljs
         [IPrintWithWriter
          (-pr-writer [x w opts]
            (-write w (helpers/keyword->prefix +registry-type+))
            (-pr-writer (proto.registry/-schemas x) w opts))]))))


(defonce ^:dynamic
  ^{:doc   "A schema registry."
    :added "0.0.2"}
  *registry*
  (registry))


(defn get-schemas
  "Returns all registered schemas. If a schema `name` is specified then return all schema versions."
  {:added "0.0.2"}
  ([]
   (proto.registry/-schemas *registry*))

  ([name]
   (proto.registry/-schemas *registry* name)))


(defn get-schema
  "Returns all versions of the schema. Returns the schema if the `version` is specified."
  {:added "0.0.2"}
  ([name]
   (proto.registry/-schema *registry* name))

  ([name version]
   (proto.registry/-schema *registry* name version)))


(defn register
  "Registers the schema in the registry."
  {:added "0.0.2"}
  [schema]
  (proto.registry/-register *registry* schema))


(defn register-predicates
  "Registers the core predicates."
  {:added "0.0.2"}
  []
  (run! register @mv/*predicate-registry))


(defn register-generator
  "Registers a custom validator and generator."
  {:added "0.0.3"}
  [validator generator]
  (mg/register validator generator))


#?(:clj
   (defmacro defschema
     "Defines and registers the schema in the registry."
     {:added "0.0.2"}
     ([schema]
      `(register ~schema))

     ([name form]
      `(register {:name ~name, :form ~form}))

     ([name doc-or-meta form]
      (let [meta (if (map? doc-or-meta) doc-or-meta {:doc doc-or-meta})]
        `(register {:name ~name, :meta ~meta, :form ~form})))

     ([name doc meta form]
      `(register {:name ~name, :meta (assoc ~meta :doc ~doc), :form ~form}))))



;;;
;; Printers
;;;;

#?(:clj
   (defmethod print-method +schema-type+ [x ^Writer w]
     (.write w ^String (helpers/keyword->prefix +schema-type+))
     (print-method (proto.schema/-unwrap x) w)))


#?(:clj
   (defmethod print-method +registry-type+ [x ^Writer w]
     (.write w ^String (helpers/keyword->prefix +registry-type+))
     (print-method (proto.registry/-schemas x) w)))



;;;;
;; Initialize default registries
;;;;

(register-predicates)
