(ns modelizer.core
  (:refer-clojure :exclude [get-validator])
  (:require
    #?(:cljs [cljs.core :refer [Atom]])
    [modelizer.helpers :as helpers]
    [modelizer.validator :as mv]
    [modelizer.generator :as mg]
    [modelizer.protocols.schema :as proto.schema]
    [modelizer.protocols.registry :as proto.registry])
  #?(:clj
     (:import
       (java.io Writer)
       (clojure.lang Atom))))

#?(:clj (set! *warn-on-reflection* true))



;;;;
;; Schema
;;;;

(def ^{:added "0.0.2", :private true}
  +schema-type+
  "A schema type."
  :modelizer/schema)


(def ^{:added "0.0.2", :dynamic true}
  *default-schema-version*
  "A default schema version."
  :schema.version/not-specified)


(def ^{:added "0.0.2", :dynamic true}
  *default-schema-status*
  "A default schema status."
  :schema.status/not-specified)


(def ^{:added "0.0.2", :dynamic true}
  *default-namespace-aliases*
  {'cljs.core      nil
   'clojure.core   nil
   'clojure.string 'string})


(def ^{:added "0.0.2", :dynamic true}
  *default-sample-generation-maximum-tries*
  "A default sample generation maximum tries."
  100)


(def ^{:added "0.0.4"}
  default-generator-options
  "A default generator options."
  {:max-tries *default-sample-generation-maximum-tries*})


;; NOTE: add default strategy and comparator for getting schema version



;;;;
;; Helper functions
;;;;

(def ^{:added "0.0.4"}
  special-forms
  "A special form types."
  #{:and :or :not
    :< :<= := :> :>= :not=
    :enum
    :coll-of})


(defn special-form?
  "Returns `true` if the given value is special form. Otherwise `false`."
  {:added "0.0.2"}
  [x]
  (and
    (sequential? x)
    (contains? special-forms (first x))))


(defn resolve-ns-alias
  "Returns a default namespace alias."
  {:added "0.0.2"}
  [ns]
  (let [alias (get *default-namespace-aliases* ns)]
    (when (contains? *default-namespace-aliases* alias)
      alias)))


(defn get-schema-alias
  "Returns a schema alias specified in the metadata."
  {:added "0.0.6"}
  [key meta]
  (get-in meta [:require key]))


(defn keyword->schema-version
  "Returns a schema alias specified in the metadata as a keyword."
  {:added "0.0.6"}
  [key meta]
  (let [name (get-schema-alias key meta)]
    [name *default-schema-version*]))


(defn vector->schema-version
  "Returns a schema alias specified in the metadata as a vector (schema name with version)."
  {:added "0.0.6"}
  [key meta]
  (let [[name version] (get-schema-alias key meta)]
    [name (or version *default-schema-version*)]))


(defn nil->schema-version
  "Returns a schema name with the default version when schema alias isn't specified in the metadata."
  {:added "0.0.6"}
  [key _]
  [key *default-schema-version*])


(defmulti resolve-schema-version
  "Returns a schema name and version."
  {:added "0.0.6"}
  (fn [key meta]
    (let [x (get-schema-alias key meta)]
      (cond
        (nil? x) :nil
        (keyword? x) :keyword
        (vector? x) :vector
        :else key))))


(defmethod resolve-schema-version :default
  [key meta]
  (let [msg (helpers/format "Unable to resolve a schema alias - the given key `%s` isn't found in the metadata" key)]
    (throw (ex-info msg {:key key, :meta meta}))))


(defmethod resolve-schema-version :keyword [key meta] (keyword->schema-version key meta))
(defmethod resolve-schema-version :vector [key meta] (vector->schema-version key meta))
(defmethod resolve-schema-version :nil [key meta] (nil->schema-version key meta))



;;;;
;; Schema generator builders
;;;;

;; TODO: Make the creation of generators and validators in a single pass?

(declare get-schema into-generator get-generator into-validator get-validator)

(defn generator->generator
  "Creates a schema generator from the given generator (clojure.test.check.generators Generator)."
  {:arglists '([g meta])
   :added    "0.0.3"}
  [g _] g)


(defn fn->generator
  "Creates a schema generator from the given function."
  {:arglists '([f meta])
   :added    "0.0.3"}
  [f _]
  (into-generator (mg/make-generator f {}) meta))


(defn var->generator
  "Creates a schema generator from the given var (clojure.lang.Var)."
  {:added "0.0.3"}
  [var meta]
  (into-generator @var meta))


(defn keyword->generator
  "Creates a schema generator from the given keyword."
  {:added "0.0.3"}
  [key meta]
  (let [[name version] (resolve-schema-version key meta)]
    (if-some [schema (get-schema name version)]
      (get-generator schema)
      (let [msg (helpers/format "Can't find schema `%s` in registry by the given alias `[%s %s]`" key name version)]
        (throw (ex-info msg {:key key, :alias [name version], :meta meta}))))))


(defmulti special-form->generator
  "Creates a schema generator from the special form."
  {:arglists '([form meta])
   :added    "0.0.3"}
  (fn [form _]
    (first form)))


(defmethod special-form->generator :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema generator from the special form - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod special-form->generator :and
  [form meta]
  (let [valid? (into-validator form meta)]
    (as-> form $
      (rest $)
      (map #(into-generator % meta) $)
      (mg/gen-one-of $)
      (mg/gen-such-that valid? $ default-generator-options))))


(defmethod special-form->generator :or
  [form meta]
  (let [valid? (into-validator form meta)]
    (as-> form $
      (rest $)
      (map #(into-generator % meta) $)
      (mg/gen-one-of $)
      (mg/gen-such-that valid? $ default-generator-options))))


(defmethod special-form->generator :not
  [form meta]
  (let [valid? (into-validator form meta)]
    (mg/gen-such-that valid? mg/gen-any? default-generator-options)))


(defmethod special-form->generator :<
  [form meta]
  (let [valid?    (into-validator form meta)
        max       (dec (second form))
        generator (mg/make-generator max {:max max})]
    (mg/gen-such-that valid? generator default-generator-options)))


(defmethod special-form->generator :<=
  [form meta]
  (let [valid?    (into-validator form meta)
        max       (second form)
        generator (mg/make-generator max {:max max})]
    (mg/gen-such-that valid? generator default-generator-options)))


(defmethod special-form->generator :=
  [form _]
  (mg/gen-return (second form)))


(defmethod special-form->generator :>
  [form meta]
  (let [valid?    (into-validator form meta)
        min       (inc (second form))
        generator (mg/make-generator min {:min min})]
    (mg/gen-such-that valid? generator default-generator-options)))


(defmethod special-form->generator :>=
  [form meta]
  (let [valid?    (into-validator form meta)
        min       (second form)
        generator (mg/make-generator min {:min min})]
    (mg/gen-such-that valid? generator default-generator-options)))


(defmethod special-form->generator :not=
  [form meta]
  (let [valid?    (into-validator form meta)
        value     (second form)
        generator (mg/make-generator value {})]
    (mg/gen-such-that valid? generator default-generator-options)))


(defmethod special-form->generator :enum
  [form _]
  (mg/gen-elements (rest form)))


(defmethod special-form->generator :coll-of
  [form meta]
  (let [[_ element & {:as opts}] form
        kind     (get opts :kind :coll)
        size     (:count opts)
        min      (or size (:min opts))
        max      (or size (:max opts))
        unique?  (get opts :distinct false)
        valid?   (into-validator form meta)
        gen-elem (into-generator element meta)
        gen-coll (mg/make-generator kind {:gen gen-elem, :kind kind, :min min, :max max, :distinct unique?, :max-tries 100})]
    (mg/gen-such-that valid? gen-coll default-generator-options)))


(defmulti into-generator
  "Creates a schema generator."
  {:arglists '([form meta])
   :added    "0.0.3"}
  (fn [form _]
    (cond
      (mg/generator? form) :generator
      (special-form? form) :special-form
      (keyword? form) :keyword
      (var? form) :var
      (fn? form) :fn
      :else form)))


(defmethod into-generator :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema generator - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod into-generator :generator [g meta] (generator->generator g meta))
(defmethod into-generator :special-form [form meta] (special-form->generator form meta))
(defmethod into-generator :keyword [key meta] (keyword->generator key meta))
(defmethod into-generator :var [var meta] (var->generator var meta))
(defmethod into-generator :fn [f meta] (fn->generator f meta))



;;;;
;; Schema validator builders
;;;;

(defn fn->validator
  "Creates a schema validator from the given function."
  {:arglists '([f meta])
   :added    "0.0.2"}
  [f _]
  (fn validator
    ([data]
     (try (f data) (catch #?(:clj Exception :cljs js/Error) _ false)))
    ([data meta]
     (try (f data meta) (catch #?(:clj Exception :cljs js/Error) _ false)))))


(defn var->validator
  "Creates a schema validator from the given var (clojure.lang.Var)."
  {:added "0.0.2"}
  [var meta]
  (into-validator @var meta))


(defn keyword->validator
  "Creates a schema validator from the given keyword."
  {:added "0.0.2"}
  [key meta]
  (let [[name version] (resolve-schema-version key meta)]
    (if-some [schema (get-schema name version)]
      (get-validator schema)
      (let [msg (helpers/format "Can't find schema `%s` in registry by the given alias `[%s %s]`" key name version)]
        (throw (ex-info msg {:key key, :alias [name version], :meta meta}))))))


(defmulti special-form->validator
  "Creates a schema validator from the special form."
  {:arglists '([form meta])
   :added    "0.0.2"}
  (fn [form _]
    (first form)))


(defmethod special-form->validator :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema validator from the special form - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod special-form->validator :and
  [form meta]
  (->> form rest (map #(into-validator % meta)) (apply every-pred)))


(defmethod special-form->validator :or
  [form meta]
  (->> form rest (map #(into-validator % meta)) (apply some-fn)))


(defmethod special-form->validator :not
  [form meta]
  (complement (into-validator (cons :or (rest form)) meta)))


(defmethod special-form->validator :<
  [form meta]
  (let [max (second form)]
    (into-validator #(< % max) meta)))


(defmethod special-form->validator :<=
  [form meta]
  (let [max (second form)]
    (into-validator #(<= % max) meta)))


(defmethod special-form->validator :=
  [form meta]
  (let [value (second form)]
    (into-validator #(= % value) meta)))


(defmethod special-form->validator :>
  [form meta]
  (let [min (second form)]
    (into-validator #(> % min) meta)))


(defmethod special-form->validator :>=
  [form meta]
  (let [min (second form)]
    (into-validator #(>= % min) meta)))


(defmethod special-form->validator :not=
  [form meta]
  (let [value (second form)]
    (into-validator #(not= % value) meta)))


(defmethod special-form->validator :enum
  [form meta]
  (let [coll (set (rest form))]
    (into-validator (partial contains? coll) meta)))


(defmethod special-form->validator :coll-of
  [form meta]
  (let [[_ element & {:as opts}] form
        kind           (get opts :kind :coll)
        size           (:count opts)
        min            (or size (:min opts))
        max            (or size (:max opts))
        unique?        (get opts :distinct false)

        kind-type?     (case kind
                         :list list?
                         :vector vector?
                         :set set?
                         :sequential sequential?
                         :coll coll?
                         (constantly false))

        kind-size?     (cond
                         (not (or min max)) (constantly true)
                         (and min max) #(<= min (count %) max)
                         min #(<= min (count %))
                         max #(<= (count %) max)
                         :else (constantly false))

        kind-elements? (partial every? (into-validator element meta))

        kind-distinct? (if-not unique?
                         (constantly true)
                         (fn [coll]
                           (cond
                             (empty? coll) true
                             (set? coll) true
                             :else (apply distinct? coll))))

        kind-coll?     (fn [coll]
                         (and (kind-type? coll)
                           (kind-size? coll)
                           (kind-elements? coll)
                           (kind-distinct? coll)))]
    (into-validator kind-coll? meta)))


(defmulti into-validator
  "Creates a schema validator."
  {:arglists '([form meta])
   :added    "0.0.2"}
  (fn [form _]
    (cond
      (special-form? form) :special-form
      (keyword? form) :keyword
      (var? form) :var
      (fn? form) :fn
      :else form)))


(defmethod into-validator :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema validator - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod into-validator :special-form [form meta] (special-form->validator form meta))
(defmethod into-validator :keyword [key meta] (keyword->validator key meta))
(defmethod into-validator :var [var meta] (var->validator var meta))
(defmethod into-validator :fn [f meta] (fn->validator f meta))



;;;;
;; Schema form builders
;;;;

(defn special-form->form [form _] form)
(defn keyword->form [key _] key)
(defn fn->form [f _] f)
(defn var->form [var _] var)


(defmulti into-form
  "Creates a schema form."
  {:arglists '([form meta])
   :added    "0.0.2"}
  (fn [form _]
    (cond
      (special-form? form) :special-form
      (keyword? form) :keyword
      (var? form) :var
      (fn? form) :fn
      :else form)))


(defmethod into-form :default
  [form meta]
  (let [type (type form)
        msg  (helpers/format "Unable to create a schema form - the given form type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :form form, :meta meta}))))


(defmethod into-form :special-form [form meta] (special-form->form form meta))
(defmethod into-form :keyword [key meta] (keyword->form key meta))
(defmethod into-form :var [var meta] (var->form var meta))
(defmethod into-form :fn [f meta] (fn->form f meta))



;;;;
;; Schema builders
;;;;

(defn with-defaults [meta]
  (cond-> meta
    (nil? (:version meta)) (assoc :version *default-schema-version*)
    (nil? (:status meta)) (assoc :status *default-schema-status*)))


(defn map->schema
  "Creates a schema from the given map."
  {:added "0.0.2"}
  [map]
  (let [name      (:name map)
        meta      (with-defaults (get map :meta {}))
        form      (into-form (:form map) meta)
        validator (into-validator form meta)
        generator (into-generator (or (:generator meta) form) meta)]
    {:name      name
     :meta      meta
     :form      form
     :validator validator
     :generator generator}))


(defn var->schema
  "Creates a schema from the given var (clojure.lang.Var)."
  {:added "0.0.2"}
  ([var]
   (let [var-meta (meta var)
         var-name (str (:name var-meta))
         var-ns   (some-> var-meta :ns ns-name symbol)
         alias    (resolve-ns-alias var-ns)
         name     (keyword (some-> alias str) var-name)]
     (var->schema name var)))

  ([name var]
   (let [var-meta  (meta var)
         var-ns    (some-> var-meta :ns ns-name symbol)
         added     (:added var-meta)
         doc       (:doc var-meta)
         meta      (with-defaults {})
         form      (into-form var meta)
         validator (into-validator form meta)
         generator (into-generator form meta)]
     (cond-> {:name      name
              :meta      meta
              :form      form
              :validator validator
              :generator generator}
       (some? doc) (assoc-in [:meta :doc] doc)
       (some? var-ns) (assoc-in [:meta :ns] var-ns)
       (some? added) (assoc-in [:meta :added] added)))))


(defmulti into-schema
  "Creates a schema."
  {:added "0.0.2"}
  (fn [schema]
    (cond
      (map? schema) :map
      (var? schema) :var
      :else schema)))


(defmethod into-schema :default
  [schema]
  (let [type (type schema)
        msg  (helpers/format "Unable to create a schema - the given type `%s` isn't implemented" type)]
    (throw (ex-info msg {:type type, :schema schema}))))


(defmethod into-schema :map [m] (map->schema m))
(defmethod into-schema :var [f] (var->schema f))


(defn schema?
  "Returns `true` if the given schema is implements `modelizer.protocols.schema/Schema` protocol.
  Otherwise `false`."
  {:added "0.0.2"}
  [schema]
  (satisfies? proto.schema/Schema schema))


(defn schema
  "Creates a schema."
  {:arglists '([schema])
   :added    "0.0.2"}
  [?schema]
  (if (schema? ?schema)
    ?schema
    (let [s         (into-schema ?schema)
          name      (:name s)
          meta      (:meta s)
          form      (:form s)
          version   (:version meta)
          status    (:status meta)
          doc       (:doc meta)
          validator (:validator s)
          generator (:generator s)]
      ^{:type +schema-type+}
      (reify proto.schema/Schema
        (-name [_] name)
        (-version [_] version)
        (-status [_] status)
        (-doc [_] doc)
        (-form [_] form)
        (-meta [_] meta)
        (-unwrap [_] s)
        (-validator [_] validator)
        (-validate [_ data] (validator data))
        (-validate [_ data meta] (validator data meta))
        (-generator [_] generator)
        (-generate [_] (mg/generate generator))
        (-generate [_ opts] (mg/generate generator opts))
        (-sample [_] (mg/sample generator))
        (-sample [_ n] (mg/sample generator n))
        (-sample [_ n opts] (mg/sample generator n opts))

        #?@(:cljs
            [IPrintWithWriter
             (-pr-writer [x w opts]
               (-write w (helpers/keyword->prefix +schema-type+))
               (-pr-writer (proto.schema/-unwrap x) w opts))])))))


(defn get-name
  "Returns a schema name."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-name schema))


(defn get-version
  "Returns a schema version."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-version schema))


(defn get-status
  "Returns a schema status."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-status schema))


(defn get-doc
  "Returns a schema doc."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-doc schema))


(defn get-form
  "Returns a schema form."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-form schema))


(defn get-meta
  "Returns a schema metadata."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-meta schema))


(defn unwrap
  "Returns a schema data."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-unwrap schema))


(defn get-validator
  "Returns a schema validator."
  {:added "0.0.2"}
  [schema]
  (proto.schema/-validator schema))


(defn validate
  "Returns `true` if the given `data` is valid for the `schema`. Otherwise `false`."
  {:added "0.0.2"}
  ([schema data]
   (proto.schema/-validate schema data))

  ([schema data meta]
   (proto.schema/-validate schema data meta)))


(defn get-generator
  "Returns a schema generator."
  {:added "0.0.3"}
  [schema]
  (proto.schema/-generator schema))


(defn generate
  "Generates a single sample value from the schema generator."
  {:added "0.0.3"}
  ([schema]
   (proto.schema/-generate schema))

  ([schema opts]
   (proto.schema/-generate schema opts)))


(defn sample
  "Generates samples from the schema generator."
  {:added "0.0.3"}
  ([schema]
   (proto.schema/-sample schema))

  ([schema n]
   (proto.schema/-sample schema n))

  ([schema n opts]
   (proto.schema/-sample schema n opts)))



;;;;
;; Registry
;;;;

(def ^{:added "0.0.2", :private true}
  +registry-type+
  "A registry type."
  :modelizer/registry)


(defn registry?
  "Returns `true` if the given schema is implements `modelizer.protocols.registry/Registry` protocol.
  Otherwise `false`."
  {:added "0.0.2"}
  [x]
  (satisfies? proto.registry/Registry x))


(defn registry
  "Creates a registry."
  {:added "0.0.2"}
  ([]
   (registry (atom {})))

  ([^Atom *registry]
   ^{:type +registry-type+}
   (reify proto.registry/Registry
     (-schemas [_] @*registry)
     (-schemas [_ name] (get @*registry name))
     (-schema [_ name] (get-in @*registry [name *default-schema-version*]))
     (-schema [_ name version] (get-in @*registry [name version]))
     (-register [_ ?schema]
       (let [s    (schema ?schema)
             path [(get-name s) (get-version s)]]
         (swap! *registry assoc-in path s)
         s))

     #?@(:cljs
         [IPrintWithWriter
          (-pr-writer [x w opts]
            (-write w (helpers/keyword->prefix +registry-type+))
            (-pr-writer (proto.registry/-schemas x) w opts))]))))


(defonce ^:dynamic
  ^{:doc   "A schema registry."
    :added "0.0.2"}
  *registry*
  (registry))


(defn get-schemas
  "Returns all registered schemas. If a schema `name` is specified then return all schema versions."
  {:added "0.0.2"}
  ([]
   (proto.registry/-schemas *registry*))

  ([name]
   (proto.registry/-schemas *registry* name)))


(defn get-schema
  "Returns all versions of the schema. Returns the schema if the `version` is specified."
  {:added "0.0.2"}
  ([name]
   (proto.registry/-schema *registry* name))

  ([name version]
   (proto.registry/-schema *registry* name version)))


(defn register
  "Registers the schema in the registry."
  {:added "0.0.2"}
  [schema]
  (proto.registry/-register *registry* schema))


(defn register-predicates
  "Registers the core predicates."
  {:added "0.0.2"}
  []
  (run! register @mv/*predicate-registry))


(defn register-generator
  "Registers a custom validator and generator."
  {:added "0.0.3"}
  [validator generator]
  (mg/register validator generator))


#?(:clj
   (defmacro defschema
     "Defines and registers the schema in the registry."
     {:added "0.0.2"}
     ([schema]
      `(register ~schema))

     ([name form]
      `(register {:name ~name, :form ~form}))

     ([name doc form]
      `(register {:name ~name, :meta {:doc ~doc}, :form ~form}))

     ([name doc meta form]
      `(register {:name ~name, :meta (assoc ~meta :doc ~doc), :form ~form}))))



;;;
;; Printers
;;;;

#?(:clj
   (defmethod print-method +schema-type+ [x ^Writer w]
     (.write w ^String (helpers/keyword->prefix +schema-type+))
     (print-method (proto.schema/-unwrap x) w)))


#?(:clj
   (defmethod print-method +registry-type+ [x ^Writer w]
     (.write w ^String (helpers/keyword->prefix +registry-type+))
     (print-method (proto.registry/-schemas x) w)))



;;;;
;; Initialize default registries
;;;;

(register-predicates)
