(ns modelizer.generator
  (:require
    [clojure.test.check.random :as random]
    [clojure.test.check.rose-tree :as rose]
    #?(:clj  [clojure.test.check.generators :as gen]
       :cljs [clojure.test.check.generators :as gen :refer [Generator]])
    [modelizer.helpers :as helpers])
  (:import
    #?@(:clj
        [(java.net URI)
         (java.util Date)
         (java.math BigDecimal)
         (clojure.test.check.generators Generator)]

        :cljs
        [(goog Uri)])))

;; TODO: rewrite generators with `requiring-resolve` and helper fns `into-inst`, `into-uri`

;;;;
;; Helper functions
;;;;

(def generator? gen/generator?)
(def gen-set gen/set)
(def gen-map gen/map)
(def gen-fmap gen/fmap)
(def gen-tuple gen/tuple)
(def gen-list gen/list)
(def gen-vector gen/vector)
(def gen-vector-distinct gen/vector-distinct)
(def gen-one-of gen/one-of)
(def gen-return gen/return)
(def gen-elements gen/elements)
(def gen-such-that gen/such-that)

(defn ->inst [x]
  #?(:clj  (Date. x)
     :cljs (js/Date. x)))


(defn ->uri [x]
  (let [s (helpers/format "http://%s.com" x)]
    #?(:clj  (URI/create s)
       :cljs (Uri. s))))


#?(:clj
   (defn ->decimal [x]
     (BigDecimal/valueOf x)))



;;;;
;; Generator builders
;;;

(defn gen-double*
  ([]
   (gen-double* nil))

  ([opts]
   (gen/double* (merge {:infinite? false, :NaN? false} opts))))


(defn gen-large-integer*
  ([]
   (gen-large-integer* nil))

  ([opts]
   (gen/large-integer* opts)))


(defn gen-coll*
  ([]
   (gen-coll* nil))

  ([{:keys [gen kind min max]}]
   (let [opts      (cond
                     (and min (= min max)) [min]
                     (and min max) [min max]
                     min [min (* 10 min)]
                     max [0 max])
         f         (case kind
                     :list (partial apply list)
                     :set set
                     identity)
         generator (apply gen-vector gen opts)]
     (gen-fmap f generator))))


(defn gen-coll-distinct*
  ([]
   (gen-coll-distinct* nil))

  ([{:keys [gen kind min max max-tries]}]
   (let [opts      {:min-elements min, :max-elements max, :max-tries max-tries}
         f         (case kind
                     :list (partial apply list)
                     :set set
                     identity)
         generator (gen-vector-distinct gen opts)]
     (gen-fmap f generator))))



;;;;
;; Generators
;;;;

(def gen-simple-type-printable? gen/simple-type-printable)
(def gen-any-printable? gen/any-printable)
(def gen-boolean? gen/boolean)
(def gen-nil? (gen-return nil))
(def gen-any? (gen-one-of [gen-nil? gen-any-printable?]))
(def gen-some? (gen-such-that some? gen-any-printable?))
(def gen-true? (gen-return true))
(def gen-false? (gen-return false))
(def gen-char? gen/char)
(def gen-string? gen/string-alphanumeric)
(def gen-double? (gen-double*))
(def gen-float? gen-double?)
(def gen-integer? gen/large-integer)
(def gen-number? (gen-one-of [gen-integer? gen-double?]))
(def gen-zero? (gen-return 0))
(def gen-even? (gen-such-that even? gen-integer?))
(def gen-odd? (gen-such-that odd? gen-integer?))
(def gen-int? gen-integer?)
(def gen-pos-int? (gen-large-integer* {:min 1}))
(def gen-neg-int? (gen-large-integer* {:max -1}))
(def gen-nat-int? (gen-large-integer* {:min 0}))
(def gen-symbol? gen/symbol-ns)
(def gen-simple-symbol? gen/symbol)
(def gen-qualified-symbol? (gen-such-that qualified-symbol? gen-symbol?))
(def gen-keyword? gen/keyword-ns)
(def gen-simple-keyword? gen/keyword)
(def gen-qualified-keyword? (gen-such-that qualified-keyword? gen-keyword?))
(def gen-ident? (gen-one-of [gen-symbol? gen-keyword?]))
(def gen-simple-ident? (gen-one-of [gen-simple-symbol? gen-simple-keyword?]))
(def gen-qualified-ident? (gen-one-of [gen-qualified-symbol? gen-qualified-keyword?]))
(def gen-list? (gen-list gen-simple-type-printable?))
(def gen-vector? (gen-vector gen-simple-type-printable?))
(def gen-map? (gen-map gen-simple-type-printable? gen-simple-type-printable?))
(def gen-set? (gen-set gen-simple-type-printable?))
(def gen-coll? (gen-one-of [gen-list? gen-vector? gen-map? gen-set?]))
(def gen-empty? (gen-elements [nil '() [] {} #{}]))
(def gen-seq? gen-list?)
(def gen-seqable? (gen-one-of [gen-nil? gen-list? gen-vector? gen-map? gen-set? gen-string?]))
(def gen-sequential? (gen-one-of [gen-list? gen-vector?]))
(def gen-associative? (gen-one-of [gen-map? gen-vector?]))
(def gen-indexed? gen-vector?)
(def gen-inst? (gen-fmap ->inst gen-integer?))
(def gen-uuid? gen/uuid)
(def gen-uri? (gen-fmap ->uri gen-uuid?))

#?(:clj (def gen-decimal? (gen-fmap ->decimal gen-double?)))
#?(:clj (def gen-ratio? gen/ratio))
#?(:clj (def gen-rational? (gen-one-of [gen-integer? gen-ratio?])))
#?(:clj (def gen-bytes? gen/bytes))



;;;;
;; Generators registry
;;;;

(defonce *generator-registry
  (atom
    {any?               gen-any?
     nil?               gen-nil?
     some?              gen-some?
     boolean?           gen-boolean?
     true?              gen-true?
     false?             gen-false?
     char?              gen-char?
     string?            gen-string?
     number?            gen-number?
     zero?              gen-zero?
     even?              gen-even?
     odd?               gen-odd?
     int?               gen-int?
     pos-int?           gen-pos-int?
     neg-int?           gen-neg-int?
     nat-int?           gen-nat-int?
     float?             gen-float?
     double?            gen-double?
     integer?           gen-integer?
     ident?             gen-ident?
     simple-ident?      gen-simple-ident?
     qualified-ident?   gen-qualified-ident?
     symbol?            gen-symbol?
     simple-symbol?     gen-simple-symbol?
     qualified-symbol?  gen-qualified-symbol?
     keyword?           gen-keyword?
     simple-keyword?    gen-simple-keyword?
     qualified-keyword? gen-qualified-keyword?
     list?              gen-list?
     vector?            gen-vector?
     map?               gen-map?
     set?               gen-set?
     ;; special-symbol?
     ;; var?
     ;; tagged-literal?
     coll?              gen-coll?
     empty?             gen-empty?
     seq?               gen-seq?
     seqable?           gen-seqable?
     sequential?        gen-sequential?
     associative?       gen-associative?
     indexed?           gen-indexed?
     ;; sorted?
     ;; counted?
     ;; reversible?
     ;; fn?
     ;; ifn?
     inst?              gen-inst?
     uuid?              gen-uuid?
     uri?               gen-uri?

     #?@(:clj
         [decimal? gen-decimal?
          ratio? gen-ratio?
          rational? gen-rational?
          bytes? gen-bytes?
          ;; class?
          ;; reader-conditional?
          ])}))


;; FIXME: add fn for register a custom generator

;;;;
;; Public API
;;;;

(defn fn->generator [f _]
  (if-some [generator (get @*generator-registry f)]
    generator
    (Generator. f)))


(defmulti make-generator
  (fn [x _]
    (cond
      (generator? x) :generator
      (keyword? x) x
      (integer? x) :integer
      (double? x) :double
      (string? x) :string
      (var? x) :var
      (fn? x) :fn
      :else x)))


(defmethod make-generator :default [_ _] gen-any?)
(defmethod make-generator :generator [g _] g)
(defmethod make-generator :integer [_ opts] (gen-large-integer* opts))
(defmethod make-generator :double [_ opts] (gen-double* opts))
(defmethod make-generator :string [_ _] gen-string?)
(defmethod make-generator :list [_ opts] (gen-coll* opts))
(defmethod make-generator :vector [_ opts] (gen-coll* opts))
(defmethod make-generator :set [_ opts] (gen-coll-distinct* opts))
(defmethod make-generator :sequential [_ opts] (gen-coll* opts))
(defmethod make-generator :coll [_ opts] (gen-coll* opts))
(defmethod make-generator :var [f opts] (make-generator @f opts))
(defmethod make-generator :fn [f opts] (fn->generator f opts))


(defn make-seed [seed]
  (if seed
    (random/make-random seed)
    (random/make-random)))


(defn register [validator generator]
  (swap! *generator-registry assoc validator generator)
  nil)


(defn generate
  ([generator]
   (generate generator nil))

  ([generator {:keys [seed size] :or {size 30}}]
   (rose/root (gen/call-gen generator (make-seed seed) size))))


(defn sample
  ([generator]
   (sample generator nil))

  ([generator n]
   (sample generator n nil))

  ([generator n {:keys [seed size] :or {size 200}}]
   (->> size
     gen/make-size-range-seq
     (map #(rose/root (gen/call-gen generator %1 %2))
       (gen/lazy-random-states (make-seed seed)))
     (take (or n 10)))))
