(ns morri.append-reports
  (:require
   [morri.append-reports.readers :as readers]
   [morri.meth450k.common
    [command-line :as cli]
    [utils :as utils :refer [tprn]]
    [db-utils :as db-utils]
    [db-reporting :as report-lib]]
   [clojure.data.csv :as csv]
   [clojure.java.io :as io])
  (:gen-class))

;; Plan is to take a csv file where the first column (or perhaps a
;; named column?) contains the database key.  Use the key to lookup
;; rows from the database and then format those rows using the
;; infrastructure from the reports system.  Then append these
;; formatted rows onto the original data and write it back out to a
;; csv file.

;; First application will be to average one or more rows from a
;; database or methylation data.  The database is created using
;; table-import.

(defn get-csv-header [file]
  (map keyword (first (utils/csv-read file))))

(defn make-header [template old-header]
  (let [new-cols (map :col template)]
    (vector (vec (map name (concat old-header new-cols))))))

(defn index-of [e coll] (first (keep-indexed #(if (= e %2) %1) coll)))

(defn get-lookup-col-index

  "Find the index of the lookup column in the header if lookup-col it is a
  name, otherwise decrement if lookup-col is a number"

  [lookup-col input-header]
  (if (number? lookup-col)
    (dec lookup-col)
    (index-of (keyword lookup-col) input-header)))

(def options-config [["-h" "--help" "Append Reports columns onto existing data"
                      :default false :flag true]
                     ["-i" "--input-file" "Input csv file"]
                     ["-k" "--lookup-col"
                      "Column in csv file to lookup in the databse, number or name"
                      :default 1]
                     ["-s" "--sqlite-file" "Sqlite file" :default false]
                     ["-m" "--mysql-db" "Mysql database, see -s" :default false]
                     ["-t" "--table" "Database table" :parse-fn keyword]
                     ["-c" "--config-file" "Report configuration, contains "]
                     ["-f" "--output-file" "File for csv output"]])

;; mysql doesn't preserve insertion order, need to insert with an auto
;; incrementing primary key, but I'm not sure how do do this with both
;; mysql and sqlite

;; configuration uses templates defined as edn-readers in
;; append-reports/templates.clj

(defn append-reports [{:keys [input-file lookup-col sqlite-file
                              mysql-db table config-file output-file]}]
  (let [config (tprn (utils/read-edn config-file readers/edn-readers))
        {:keys [template index-col]} config
        input-header (get-csv-header input-file)
        key-index (get-lookup-col-index lookup-col input-header)
        output-header (make-header template input-header)
        input-data (drop 1 (utils/csv-read input-file))
        limit-probes (map #(% key-index) input-data)
        db-spec (cond sqlite-file (db-utils/sqlite-db sqlite-file)
                      mysql-db (db-utils/mysql-db mysql-db)
                      :else (throw (Exception. "Define either --sqlite-file or --mysql-db")))
        db-pool (db-utils/make-delayed-db-pool db-spec)
        formatted-results (report-lib/db-report
                           @db-pool table template limit-probes index-col)
        output-data (remove nil? (map #(when (and %1 %2) (concat %1 %2))
                                      input-data formatted-results))]
    (with-open [wrtr (io/writer output-file)]
      (csv/write-csv wrtr output-header)
      (csv/write-csv wrtr output-data))))

(defn -main
  "I parse the command line arguments and pass them on."
  [& args]
  (let [options (cli/parse-command-line args options-config)]
    (append-reports options)))
