(ns morri.lib.def-cache
  (:require [clojure.core.cache :as cache :refer [defcache]]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [taoensso.nippy :as nippy]
            [clojure.repl :refer :all]
            [clojure.set :as set]
            [loom.alg :as graph-alg]
            [loom.graph :as graph]
            [loom.io :as loom-io]
            [clojure.pprint :refer [pprint]])
  (:import [clojure.core.cache CacheProtocol]
           [java.io PushbackReader DataInputStream DataOutputStream]))

;; Possible Enhancements
;; 1. Work around the anonymous function problem
;; 2. Use a future when saving the cache
;; 3. When saving the cache, only save what needs to be updated.  Read
;; the saved cache file and compare keys before saving an
;; item. Perhaps have two functions for this.  Keep
;; save-cache/save-sesion and add a function update-saved-session.
;; 4. Rename save-cache to save-session. Same for load-cache.
;; 5. Rename cache-file to cache-index
;; 6. Closures also fool the cache for the obvious reason that the
;; function can close over data that has changed, but since the form
;; is the same, then the cache won't update.  Can I fix this by
;; storing the hash_value of each dependency?  Will the hash change
;; each time the function is updated?
;; 7. Local variables like inside a let will be recorded as
;; dependencies even though they are local.  This only introduces
;; false positive dependencies but it could perhaps create circular
;; dependencies.  We could filter the deps list to only include vars
;; already cached.  This should significantly reduce the number of
;; dependencies being dealt with.  Local variables that shadow
;; external variables would still be a problem in this scheme.
;; 8. To get around the closure issue with functions wrapping around
;; closures obscuring the dependencies, we could create a new macro
;; for a cached function.  For functions, we would just be caching the
;; key and form for dependency managment.  In this way, a variable
;; could depend on a function and the function could depend on other
;; vars allowing us to see the dependency flow more clearly.

(def default-cache-dir "def-cache")

(def default-cache-file "def-cache")

;; cache is {{:ns-keyword ns-keyword :deps deps} data ... }

(defn tprn [x] (prn x) x)

(defn ns-symb-keys->cache-keys [cache]
  "Takes the keys of the cache and returns a map of ns-symb-keys to
  cache-keys"
  (let [cache-keys (keys cache)
        ns-symb-keys (map :ns-symb-key cache-keys)]
    (zipmap ns-symb-keys cache-keys)))

(defcache DefCache [cache]
  CacheProtocol
  (lookup [_ cache-key]
          (cache cache-key))
  (lookup [_ cache-key not-found]
          (get cache cache-key not-found))
  (has? [this cache-key]
        (contains? cache cache-key))
  (hit [this cache-key] this)
  (evict [_ item]
         (let [cache-key (if (keyword? item)
                           ((ns-symb-keys->cache-keys cache) item)
                           item)]
           (DefCache. (dissoc cache cache-key))))
  (miss [_ cache-key value]
        (let [ns-symb-key (:ns-symb-key cache-key)
              old-cache-key ((ns-symb-keys->cache-keys cache)
                             (:ns-symb-key cache-key))]
          (DefCache.
            (assoc (dissoc cache old-cache-key) cache-key value))))
  (seed [_ base]
        (DefCache. base))
  Object
  (toString [_] (str cache)))

(def empty-def-cache (DefCache. {}))

(def the-cache (atom empty-def-cache))

(defn reset-def-cache []
  (reset! the-cache empty-def-cache))

(defn serialize-to-file [file-path data]
  (println "Saving" file-path)
  (with-open [w (io/output-stream file-path)]
    (nippy/freeze-to-out! (DataOutputStream. w) data)))

(defn deserialize-from-file [file-path]
  (println "Loading" file-path)
  (with-open [r (io/input-stream file-path)]
    (nippy/thaw-from-in! (DataInputStream. r))))

(defn cache-file [cache-dir]
  (io/file cache-dir default-cache-file))

(defn ensure-dir [a-dir]
  (when-not (.isDirectory (io/file a-dir))
    (.mkdirs (io/file a-dir))))

(defn save-cache
  ([] (save-cache default-cache-dir))
  ([cache-dir]
     (println "Saving the Cache")
     (let [cache-for-saving
           (into {}
                 (for [[k v] @the-cache
                       :let [ns-symb-key (:ns-symb-key k)
                             key-ns (namespace ns-symb-key)
                             symb-name (name ns-symb-key)
                             file-name (io/file cache-dir key-ns symb-name)]]
                   {k file-name}))]
       (ensure-dir cache-dir)
       (doseq [[k v] @the-cache
               :let [file-name (cache-for-saving k)
                     ns-dir (.getParent file-name)]]
         (ensure-dir ns-dir)
         (serialize-to-file file-name v))
       (serialize-to-file (cache-file cache-dir) cache-for-saving))
     (println "Cache Saved")))

(defn load-cache
  ([] (load-cache default-cache-dir))
  ([cache-dir]
     (println "Loading the Cache")
     (let [saved-cache-file (deserialize-from-file (cache-file cache-dir))
           load-fn (fn [cache [k file-name]]
                     (assoc! cache k (deserialize-from-file file-name)))
           saved-cache (persistent!
                        (reduce load-fn (transient {}) saved-cache-file))]
       (reset! the-cache (cache/seed empty-def-cache saved-cache))
       (println "Cache loaded"))))

(defn symbol->ns-symb-key [symb]
  (when (symbol? symb)
    (if-let [the-var (resolve symb)]
      (keyword (str (ns-name (:ns (meta the-var)))) (str symb))
      (keyword (str *ns*) (str symb)))))

(defn form->ns-symb-key-list [form]
  (cond
   (symbol? form)
   (symbol->ns-symb-key form)
   (coll? form)
   (map form->ns-symb-key-list (flatten (seq form))))) 

(defn make-dep-graph [cache]
  (let [cache-keys (keys cache)
        nodes (map :ns-symb-key cache-keys)
        deps (map :deps cache-keys)
        adjacency-map (zipmap nodes deps)]
    (when-not (empty? adjacency-map)
      (graph/transpose (graph/digraph adjacency-map)))))

(defn successors [cache ns-symb-key]
  "Takes our cache, builds a dependency graph and returns the
  successors of a particular symbol"
  (when-let [dep-graph (make-dep-graph cache)]
    (graph/successors dep-graph ns-symb-key)))

(defn view-dep-graph []
  (when-let [dep-graph (make-dep-graph @the-cache)]
    (loom-io/view dep-graph)))

;; useful graph functions
;; (dag? g)
;; (loom-io/view g)
;; (pprint (graph-alg/topsort g))

(defn remove-successors [cache ns-symb-key]
  (let [succ (successors cache ns-symb-key)]
    (apply dissoc cache succ)))

;; Note def-cached doesn't work with anonymous functions because they
;; get replaced in the form by random symbols before the macro gets to
;; see the form.  Can't think of a good fix for this yet.

(defn form->symb-set [form]
  (apply hash-set
         (remove nil?
          (flatten 
           (form->ns-symb-key-list form)))))

(defn cached-symb-set [cache]
  (apply hash-set (map :ns-symb-key (keys @the-cache))))

(defmacro def-cached [symb form]
  (let [ns-symb-key (symbol->ns-symb-key symb)
        cache-key {:ns-symb-key ns-symb-key
                   :form (str form)}
        potential-deps (form->symb-set form)]
    ;; We wait till runtime to compute the actual deps using the
    ;; runtime cache keys
    `(let [runtime-cache-key# (assoc ~cache-key
                                :deps (set/intersection
                                       (cached-symb-set @the-cache)
                                       ~potential-deps))]
       (if (contains? @the-cache runtime-cache-key#)
        (def ~symb (get @the-cache runtime-cache-key#))
        (do 
          (swap! the-cache remove-successors ~ns-symb-key)
          (def ~symb (get
                      (swap! the-cache assoc runtime-cache-key# ~form)
                      runtime-cache-key#)))))))

(defmacro clear-cached-def [symb]
  (let [ns-symb-key (symbol->ns-symb-key symb)]
    `(do
       (remove-successors @the-cache ~ns-symb-key)
       (swap! the-cache dissoc ~ns-symb-key)
       (println "Removed" ~ns-symb-key))))

(defn long-computation [answer]
  (println "Running a long computation")
  (Thread/sleep 1000)
  answer)

;; (def-cached t1 (long-computation 10))
;; (def-cached t1 "test-value")
;; (def-cached t1 (long-computation 20))

;; Setup some dependent defs
;; (def-cached t1 10)
;; (def-cached t2 (inc t1))
;; (def-cached t3 (* t1 t2))
;; (def-cached t2 20)
;; (def-cached t3 {:a 1 [2 3] t1})
;; (pprint (topo-order @the-cache))

;; (clear-cached-def t1)

;; (reset-def-cache)
;; @the-cache


