(ns morri.lib.def-cache
  (:require [clojure.core.cache :as cache :refer [defcache]]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [taoensso.nippy :as nippy]
            [clojure.repl :refer :all]
            [clojure.set :as set]
            [loom.alg :as graph-alg]
            [loom.graph :as graph]
            [loom.io :as loom-io]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str])
  (:import [clojure.core.cache CacheProtocol]
           [java.io PushbackReader DataInputStream DataOutputStream]))

;; Possible Enhancements
;; Done -- 1. Work around the anonymous function problem
;; Probably we can just pull out any gensyms when we save the form
;; What is a regular expression for a gensym? Probably I can just
;; strip out between __ and #
;; ? 2. Use a future when saving the cache
;; 3. When saving the cache, only save what needs to be updated.  Read
;; the saved cache file and compare keys before saving an
;; item. Perhaps have two functions for this.  Keep
;; save-cache/save-sesion and add a function update-saved-session.
;; Done--4. Rename save-cache to save-session. Same for load-cache.
;; Done--5. Rename cache-file to cache-index
;; Done--6. Closures also fool the cache for the obvious reason that the
;; function can close over data that has changed, but since the form
;; is the same, then the cache won't update.  Can I fix this by
;; storing the hash_value of each dependency?  Will the hash change
;; each time the function is updated?
;; Done--7. Local variables like inside a let will be recorded as
;; dependencies even though they are local.  This only introduces
;; false positive dependencies but it could perhaps create circular
;; dependencies.  We could filter the deps list to only include vars
;; already cached.  This should significantly reduce the number of
;; dependencies being dealt with.  Local variables that shadow
;; external variables would still be a problem in this scheme.
;; Done --8. To get around the closure issue with functions wrapping around
;; closures obscuring the dependencies, we could create a new macro
;; for a cached function.  For functions, we would just be caching the
;; key and form for dependency managment.  In this way, a variable
;; could depend on a function and the function could depend on other
;; vars allowing us to see the dependency flow more clearly.  This can
;; probably be fixed by switching to clojure analyzer
;; 9. Switch to closure analyzer for more precise resolution of
;; dependencies.  Currently internal dependencies in a let or other
;; binding form can be improperly recorded as real dependencies.
;; e.g. a will be recorded as a dependency of b in this example
;; (def-cached a 10)
;; (def-cached b (let [a 20] (inc a)))
;; Done--10. Is it a problem that we dereference the same atom multiple
;; times in some of our functions?  What would happen if somebody else
;; changed our atom between dereferences?  To fix this we should turn
;; our macro body into a single swap call that takes the cache,
;; cache-key and potential-deps.
;; 11. Rename cache-index, session-index

(def default-session-dir "def-cache")

(def default-session-index "def-session-index")

;; cache is {{:ns-keyword ns-keyword :deps deps} data ... }

(defn tprn [x] (prn x) x)

(defn ns-symb-keys->cache-keys [cache]
  "Takes the keys of the cache and returns a map of ns-symb-keys to
  cache-keys"
  (let [cache-keys (keys cache)
        ns-symb-keys (map :ns-symb-key cache-keys)]
    (zipmap ns-symb-keys cache-keys)))

(defn item->cache-key [cache item]
  "Let's us look up items in the cache by simple ns-keyword-key or by
  the full key"
  (if (keyword? item)
    ((ns-symb-keys->cache-keys cache) item)
    item))

(defcache DefCache [cache]
  CacheProtocol
  (lookup [_ item]
          (let [cache-key (item->cache-key cache item)]
            (cache cache-key)))
  (lookup [_ item not-found]
          (let [cache-key (item->cache-key cache item)]
            (get cache cache-key not-found)))
  (has? [_ item]
        (let [cache-key (item->cache-key cache item)]
          (contains? cache cache-key)))
  (hit [this _] this)
  (evict [_ item]
         (let [cache-key (item->cache-key cache item)]
           (DefCache. (dissoc cache cache-key))))
  (miss [_ cache-key value]
        (let [ns-symb-key (:ns-symb-key cache-key)
              old-cache-key ((ns-symb-keys->cache-keys cache)
                             (:ns-symb-key cache-key))]
          (DefCache.
            (assoc (dissoc cache old-cache-key) cache-key value))))
  (seed [_ base]
        (DefCache. base))
  Object
  (toString [_] (str cache)))

(def empty-def-cache (DefCache. {}))

(def the-cache (atom empty-def-cache))

(defn reset-def-cache []
  (reset! the-cache empty-def-cache))

(defn serialize-to-file [file-path data]
  (println "Saving" file-path)
  (with-open [w (io/output-stream file-path)]
    (nippy/freeze-to-out! (DataOutputStream. w) data)))

(defn deserialize-from-file [file-path]
  (println "Loading" file-path)
  (with-open [r (io/input-stream file-path)]
    (nippy/thaw-from-in! (DataInputStream. r))))

(defn session-index-file [session-dir]
  (io/file session-dir default-session-index))

(defn ensure-dir [a-dir]
  (when-not (.isDirectory (io/file a-dir))
    (.mkdirs (io/file a-dir))))

(defn make-session-index [cache session-dir]
  (into {}
        (for [[k v] cache
              :let [ns-symb-key (:ns-symb-key k)
                    key-ns (namespace ns-symb-key)
                    symb-name (name ns-symb-key)
                    file-name (io/file session-dir key-ns symb-name)]]
          {k file-name})))

(defn save-session
  ([] (save-session default-session-dir))
  ([session-dir]
     (println "Saving the Cache in" session-dir)
     (let [cache @the-cache
           session-index (make-session-index cache session-dir)]
       (ensure-dir session-dir)
       (doseq [[k v] cache
               :let [file-name (session-index k)
                     ns-dir (.getParent file-name)]]
         (ensure-dir ns-dir)
         (serialize-to-file file-name v))
       (serialize-to-file (session-index-file session-dir) session-index))
     (println "Cache Saved")))

(defn update-saved-session
  ([] (update-saved-session default-session-dir))
  ([session-dir]
     (println "Updating the saved cache in" session-dir)
     (if-let [old-session-index (deserialize-from-file
                               (session-index-file session-dir))]
       (let [cache @the-cache
             old-cache-key-set (apply hash-set (keys old-session-index))
             session-index (make-session-index cache session-dir)]
         (doseq [[k v] cache
                 :when (not (old-cache-key-set k))
                 :let [file-name (session-index k)
                       ns-dir (.getParent file-name)]]
           (ensure-dir ns-dir)
           (serialize-to-file file-name v))
         (serialize-to-file (session-index-file session-dir) session-index)
         (println "Cache updated"))
       (save-session session-dir))))

(defn load-session
  ([] (load-session default-session-dir))
  ([session-dir]
     (println "Loading the Cache")
     (let [saved-session-index (deserialize-from-file
                                (session-index-file session-dir))
           load-fn (fn [cache [k file-name]]
                     (assoc! cache k (deserialize-from-file file-name)))
           saved-cache (persistent!
                        (reduce load-fn (transient {}) saved-session-index))]
       (reset! the-cache (cache/seed empty-def-cache saved-cache))
       (println "Cache loaded"))))

(defn symbol->ns-symb-key [symb]
  (when (symbol? symb)
    (if-let [the-var (resolve symb)]
      (let [m (meta the-var)]
        (keyword (str (ns-name (:ns m))) (str (:name m))))
      (keyword (str *ns*) (str symb)))))

(defn form->ns-symb-key-list [form]
  (cond
   (symbol? form)
   (symbol->ns-symb-key form)
   (coll? form)
   (map form->ns-symb-key-list (flatten (seq form))))) 

(defn make-dep-graph [cache]
  (let [cache-keys (keys cache)
        nodes (map :ns-symb-key cache-keys)
        deps (map :deps cache-keys)
        adjacency-map (zipmap nodes deps)]
    (when-not (empty? adjacency-map)
      (graph/transpose (graph/digraph adjacency-map)))))

(defn successors [cache ns-symb-key]
  "Takes our cache, builds a dependency graph and returns the
  successors of a particular symbol"
  (when-let [dep-graph (make-dep-graph cache)]
    (graph/successors dep-graph ns-symb-key)))

(defn view-dep-graph []
  (when-let [dep-graph (make-dep-graph @the-cache)]
    (loom-io/view dep-graph)))

;; useful graph functions
;; (dag? g)
;; (loom-io/view g)
;; (pprint (graph-alg/topsort g))

(defn remove-with-successors [cache ns-symb-key]
  (let [succ (successors cache ns-symb-key)]
    (apply dissoc cache ns-symb-key succ)))

;; Note def-cached doesn't work with anonymous functions because they
;; get replaced in the form by random symbols before the macro gets to
;; see the form.  Can't think of a good fix for this yet.

(defn form->symb-set [form]
  (apply hash-set
         (remove nil?
          (flatten 
           (form->ns-symb-key-list form)))))

(defn cached-symb-set [cache]
  (apply hash-set (map :ns-symb-key (keys cache))))

(defn runtime-deps [cache compile-time-deps]
  (set/intersection
   (cached-symb-set cache)
   compile-time-deps))

(defn make-runtime-cache-key [cache compile-time-cache-key compile-time-deps]
  (let [ns-symb-key (:ns-symb-key compile-time-cache-key)]
    (assoc compile-time-cache-key
      :deps
      (disj
       (runtime-deps cache compile-time-deps)
       ns-symb-key               ;we don't want to depend on ourselves
       ))))

(defn strip-out-gensym [s]
  (str/replace s #"__[^#]*#" ""))

(comment
  (strip-out-gensym "(def-cached t-anon (fn* [p1__9398#] (inc p1__9398#)))"))

(defn update-cache [cache cache-key potential-deps delayed-value]
  (let [runtime-cache-key (make-runtime-cache-key cache cache-key potential-deps)]
    (if (contains? cache runtime-cache-key)
      cache
      (let [updated-cache (remove-with-successors cache (:ns-symb-key cache-key))
            updated-cache-key (make-runtime-cache-key
                               updated-cache cache-key potential-deps)]
        (assoc updated-cache updated-cache-key @delayed-value)))))

(defmacro def-cached [symb form]
  (let [ns-symb-key (symbol->ns-symb-key symb)
        str-form (strip-out-gensym (str form))
        cache-key {:ns-symb-key ns-symb-key
                   :form str-form}
        potential-deps (form->symb-set form)]
    ;; We wait till runtime to compute the actual deps using the
    ;; actual cached defs
    `(let [delayed-value# (delay ~form)]
       (def ~symb
         (get
          (swap! the-cache update-cache ~cache-key ~potential-deps delayed-value#)
          ~ns-symb-key)))))

(defmacro defn-cached [symb binding & forms]
  (let [ns-symb-key (symbol->ns-symb-key symb)
        str-forms (strip-out-gensym (str forms))
        cache-key {:ns-symb-key ns-symb-key
                   :form str-forms}
        bound-vars (form->symb-set binding)
        potential-deps (set/difference
                        (form->symb-set forms)
                        bound-vars)]
    `(do
       (swap! the-cache update-cache ~cache-key ~potential-deps (delay :function))
       (defn ~symb ~binding ~@forms))))

(defmacro clear-cached-def [symb]
  (let [ns-symb-key (symbol->ns-symb-key symb)]
    `(do
       (swap! the-cache remove-with-successors ~ns-symb-key)
       (println "Removed" ~ns-symb-key))))

(comment
  (defn long-computation [answer]
   (println "Running a long computation")
   (Thread/sleep 1000)
   answer)
  )

;; (def-cached t1 (long-computation 10))
;; (def-cached t1 "test-value")
;; (def-cached t1 (long-computation 20))

;; (def-cached a (long-computation (#(inc %) 1)))
;; (def-cached a (long-computation (#(inc %) 1)))

;; Setup some dependent defs
;; (def-cached t1 10)
;; (def-cached t2 (inc t1))
;; (def-cached t3 (* t1 t2))
;; (pprint @the-cache)
;; (def-cached t2 20)
;; (def-cached t3 {:a 1 [2 3] t1})
;; (pprint (topo-order @the-cache))

;; (def-cached t1 100)
;; (defn-cached t2 [a] (+ a t1))
;; (def-cached t3 (t2 10))

;; (def-cached t-anon #(inc %))

;; (clear-cached-def t1)

;; ()

(comment
 (defn-cached tf [{:keys [a b]}]
   (println "Running the function")
   (+ a b))
 )

;; (tf {:a 1 :b 2})

;; Trying to make a circular dependency
;; (def-cached t1 (let [t2 1] (inc t2)))
;; (def-cached t2 (inc t1))
;; This doesn't work because when we define t1, t2 is not present so
;; it is not considered as a dependency

;; Trying again to make a circular dependency
;; (def-cached a 10)
;; (def-cached b (let [a 20] (inc a)))
;; (def-cached c (+ a b))
;; This creates an incorrect dependency of b on a, but not a circle
;; The incorrect dependencies don't seem good, but I don't think they
;; are causing any serious problems

;; (def-cached a (let [b 10] (inc b)))
;; (def-cached b (let [a 10] (inc a)))
;; (def-cached a (let [b 10] (inc b)))
;; This is also OK because the deps change on the second invocation
;; of (def-cached a ...), so the cache key is updated

;; (def-cached c 10)
;; (def-cached c (let [c 1] (inc c)))
;; This one creates a problem where c depends on itself.
;; To fix this we should remove the item and its successors first,
;; before we calculate dependencies.  I have a feeling that since the
;; order of removal and dependency is wrong we might have other
;; problems as well.

;; (def-cached a 1)
;; (def-cached b 2)
;; (save-session)
;; (def-cached b 3)
;; This should only print saving "b"
;; (update-saved-session)

;; (reset-def-cache)
;; (pprint @the-cache)



