(ns morri.meth-bed
  (:require
   [morri.meth450k.common
    [db-utils :as db-utils]
    [utils :as utils :refer [tprn]]
    [db-reporting :as reporting]
    [command-line :as cli]]
   [morri.meth-bed.bed-templates :as bed-templates]
   [clojure.java.io :as io]
   [clojure.string :as string])
  (:import [java.io File])
  (:gen-class))

;;;; These functions have the bed-type specific code
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn make-header-opts [bed-type]
  (case bed-type
    :abs-meth [:viewLimits "0:1" :autoScale "off"]
    :std-dev [:viewLimits "0:0.1" :autoScale "off" :yLineOnOff "off"]
    :diff-meth [:viewLimits "-0.1:0.1" :autoScale "off" :yLineOnOff "off"]))

(defn make-bedgraph-template [bed-type beta-cols]
  (case bed-type
    :abs-meth (bed-templates/avg-bedgraph-template beta-cols)
    :std-dev (bed-templates/std-dev-bedgraph-template beta-cols)
    :diff-meth (bed-templates/diff-bedgraph-template beta-cols)))

(defn make-track-string [bed-type track]
  (case bed-type
    (:abs-meth :std-dev) (str (name track) "_" (name bed-type))
    :diff-meth (str
                (name (first track))
                "_vs_"
                (name (second track))
                "_"
                (name bed-type))))

(defn get-beta-cols [bed-type track config]
  (case bed-type
    (:abs-meth :std-dev) (get-in config [:groups track])
    :diff-meth (map #(get-in config [:groups %]) track)))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn make-bedgraph-filename [sqlite-file meth-string]
  (.getPath (File. (utils/dir-name sqlite-file) (str meth-string ".bed"))))

(defn write-bedgraph [file-name template header config limit-probes]
  (let [{:keys [sqlite-file table index-col]} config
        db-spec (db-utils/sqlite-db sqlite-file)
        db (if limit-probes
             (db-utils/make-delayed-db-pool)
             (delay db-spec))
        bed-data (reporting/db-report @db table template limit-probes index-col)]
    (with-open [wrtr (io/writer file-name)]
      (.write wrtr header)
      (doseq [row bed-data]
        (when (not-any? nil? row)
          (.write wrtr (str (string/join \tab row) \newline)))))
    file-name))

(defn make-bedgraphs [bed-type config limit-probes]
  (for [track (bed-type config)
        :let [beta-cols (get-beta-cols bed-type track config)
              track-string (make-track-string bed-type track)
              bedgraph-file (make-bedgraph-filename
                             (:sqlite-file config)
                             track-string)
              header-opts (make-header-opts bed-type)
              bedgraph-header (apply bed-templates/make-bedgraph-header
                                     track-string track-string header-opts)
              bedgraph-template (make-bedgraph-template bed-type beta-cols)]]
    (do
      (println "Making" bed-type "beds for: " beta-cols)
      (write-bedgraph
       bedgraph-file bedgraph-template bedgraph-header config limit-probes))))

(def options-config
  [["-h" "--help" "Generate ucsc tracks for 450K data"
    :default false :flag true]
   ["-c" "--config-file" "edn configuration file for track generation"]
   ["-s" "--sqlite-file" "File of the sqlite database"]
   ["-t" "--table" "Table to use in sqlite database" :parse-fn keyword]
   ["-l" "--limit-probes" "Limit to this list of probes, see index-col"
    :default false]
   ["-idx" "--index-col" "limit-probes lookup in this db column"
    :default false :parse-fn keyword]
   ["--abs-meth" "Generate abs meth tracks" :flag true :default false]
   ["--diff-meth" "Generate diff meth tracks" :flag true :default false]
   ["--std-dev" "Generate std dev tracks" :flag true :default false]
   ["--prefix" "Prefix for the multi-track .gz files" :defualt false
    :parse-fn #(if % (str % "_") "")]])

(defn make-beds [cl-opts]
  (let [config (tprn (merge-with #(when %1 %2)
                                 cl-opts
                                 (utils/read-edn (:config-file cl-opts))))
        limit-probe-filename (:limit-probes config)
        limit-probes (when limit-probe-filename
                       (string/split-lines (slurp limit-probe-filename)))]
    (doseq [bed-type [:abs-meth :diff-meth :std-dev]]
      (when (bed-type config)
        (println "Making" bed-type "beds")
        (utils/zip-cat (str (:prefix config) (name bed-type) ".bed")
                       (make-bedgraphs bed-type config limit-probes))))))

(defn -main
  "I don't do a whole lot ... yet."
  [& args]
  (let [options (cli/parse-command-line args options-config)]
    (make-beds options)))
