(ns dead.dead3
  #?(:clj
     (:import [java.nio ByteBuffer]))
  (:require [criterium.core :as c]))


(def pretty #(binding [clojure.pprint/*print-miser-width* 100
                       clojure.pprint/*print-right-margin* 100]
              (clojure.pprint/pprint %)))
(def mm #(pretty (macroexpand %)))
(def cc #(c/with-progress-reporting (c/quick-bench %)))

(set! *warn-on-reflection* true)

;;;; ============================================================================================
;;;; Util

(defn bimap [items]
  (->> items
       (map-indexed #(vector (- %1 32768) %2))
       (mapcat (fn [[a b]] [[a b] [b a]]))
       (into {})))

(defn primitive-size [type]
  (case type
    ::byte 1
    ::short 2
    ::int 4
    ::float 4
    ::double 8
    ::char 2
    ::boolean 1))

(defn short->bytes [x]
  (#?(:clj  (partial into-array Byte/TYPE)
      :cljs clj->js)
    [(-> x (bit-shift-right 8) (bit-and 0xFF) (unchecked-byte))
     (-> x (bit-and 0xFF) (unchecked-byte))]))

;;;; ============================================================================================
;;;; Buffers

(defprotocol AutoByteBuffer
  (take! [this type position])
  (put! [this data type position]))

(do #?(:clj  (extend-type ByteBuffer
               AutoByteBuffer
               (take! [this type position]
                 (let [p @position]
                   (swap! p (+ p (primitive-size type)))
                   (case type
                     ::byte (.get this ^int p)
                     ::short (.getShort this p)
                     ::int (.getInt this p)
                     ::float (.getFloat this p)
                     ::double (.getDouble this p)
                     ::char (.getChar this p)
                     ::boolean (-> this
                                   (.get ^int (quot p 8))
                                   (bit-test (rem p 8)))
                     nil)))
               (put! [this data type position]
                 (let [p @position]
                   (swap! position (+ p (primitive-size type)))
                   (case type
                     ::byte (.put this (byte data) p)
                     ::short (.putShort this (short data) p)
                     ::int (.putInt this (int data) p)
                     ::float (.putFloat this (float data) p)
                     ::double (.putDouble this (double data) p)
                     ::char (.putChar this (char data) p)
                     ::boolean (.put this (-> this
                                              (.get ^int (quot p 8))
                                              (bit-set (rem p 8))
                                              (unchecked-byte)))
                     nil))))

       :cljs (extend-type js/DataView
               AutoByteBuffer
               (take! [this type position]
                 (let [p @position]
                   (swap! position (+ p (primitive-size type)))
                   (case type
                     ::byte (.getInt8 this p)
                     ::short (.getInt16 this p)
                     ::int (.getInt32 this p)
                     ::float (.getFloat32 this p)
                     ::double (.getFloat64 this p)
                     ::char (.getUint8 this p)
                     ::boolean (-> this
                                   (.getInt8 (quot p 8))
                                   (bit-test (rem p 8)))
                     nil)))
               (put! [this data type position]
                 (let [p @position]
                   (swap! position (+ p (primitive-size type)))
                   (case type
                     ::byte (.setInt8 this p data)
                     ::short (.setInt16 this p data)
                     ::int (.setInt32 this p data)
                     ::float (.setFloat32 this p data)
                     ::double (.setFloat64 this p data)
                     ::char (.setUint8 this p data)
                     ::boolean (.setInt8 this (-> this
                                                  (.getInt8 (quot p 8))
                                                  (bit-set (rem p 8))
                                                  (unchecked-byte)))
                     nil))))))

(defn reset-wbuffer! [wbuffer]
  (assoc wbuffer :bit-pos 0
                 :byte-pos (:max-bits wbuffer)))

(defn make-wbuffer [max-bits max-bytes]
  (let [length (+ max-bits max-bytes)]
    {:buffer        #?(:clj  (ByteBuffer/allocate length)
                       :cljs (js/DataView. (js/ArrayBuffer. length)))
     :bit-position  (atom 0)
     :byte-position (atom max-bits)
     :max-bits      max-bits}))

(defn wrap-bytes [bytes]
  (let [buffer #?(:clj (ByteBuffer/wrap bytes)
                  :cljs (js/DataView. bytes))
        schema-code    (take! buffer ::short (atom 0))
        bit-length     (take! buffer ::short (atom 2))]
    [{:buffer        buffer
      :bit-position  (atom 32)
      :byte-position (atom (+ 4 bit-length))}
     schema-code]))

(defn unwrap-wbuffer [{:keys [buffer bit-position byte-position max-bits]}]
  (let [bits-length  (int (* 8 (Math/ceil (/ @bit-position 8))))
        bytes-length (- @byte-position max-bits)
        total-length (+ 2 bits-length bytes-length)]
    #?(:clj  (let [buffer-bytes (.array ^ByteBuffer buffer)
                   bytes        (byte-array total-length)]
               (System/arraycopy (short->bytes bits-length) 0 buffer-bytes 0 2)
               (System/arraycopy buffer-bytes 0 bytes 2 bits-length)
               (System/arraycopy buffer-bytes max-bits bytes (+ 2 bits-length) bytes-length)
               bytes)
       :cljs (let [buffer-bytes (.-buffer buffer)
                   bytes        (js/Int8Array. total-length)]
               (.set bytes (short->bytes bits-length) 0)
               (.set bytes (js/Int8Array. (.slice buffer-bytes 0 bits-length)) 2)
               (.set bytes (js/Int8Array. (.slice buffer-bytes max-bits (+ max-bits bytes-length))) (+ 2 bits-length))
               (.-buffer bytes)))))

;;;; ============================================================================================
;;;; De/serialization

(def primitives
  #{::byte ::short ::int ::float ::double ::char})

(defn primitive? [type]
  (primitives type))

(def composites
  #{::list ::vector ::set ::map ::tuple ::record})

(defn composite? [schema]
  (and (vector? schema)
       (composites (first schema))))

(defn disj-composite [[a b & more]]
  (let [u  a
        v  (if (map? b) b {})
        w  (vec (if (or (map? b) (nil? b))
                  more
                  (cons b more)))
        w1 (first w)]
    (if (and (sequential? w1)
             (not (composite? w1)))
      [u v w1]
      [u v w])))


(declare serialize-1)
(declare deserialize-1)


(defn serialize-primitive [schema _ {:keys [buffer bit-position byte-position]} data]
  (let [position (if (= ::boolean schema) bit-position byte-position)]
    `(put! ~buffer ~data ~schema ~position)))

(defn deserialize-primitive [schema _ {:keys [buffer bit-position byte-position]}]
  (let [position (if (= ::boolean schema) bit-position byte-position)]
    `(take! ~buffer ~schema ~position)))


(defn serialize-string [_ _ {:keys [buffer byte-position]} data]
  (let [char (gensym "char_")]
    `(do (put! ~buffer (count ~data) ::short ~byte-position)
         (run! (fn [~char] (put! ~buffer ~char ::char ~byte-position))
               ~data))))

(defn deserialize-string [_ _ {:keys [buffer byte-position]}]
  `(->> (take! ~buffer ::short ~byte-position)
        (repeatedly (fn [] (take! ~buffer ::char ~byte-position)))
        (doall)
        (apply str)))


(defn serialize-keyword [_ {:keys [keyword-map]} {:keys [buffer byte-position]} data]
  `(put! ~buffer (get ~keyword-map ~data) ::short ~byte-position))

(defn deserialize-keyword [_ {:keys [keyword-map]} {:keys [buffer byte-position]}]
  `(get ~keyword-map (take! ~buffer ::short ~byte-position)))


(defn serialize-coll [schema config {:keys [buffer byte-position] :as wbuffer} data]
  (let [[_ _ sub-schema] (disj-composite schema)
        coll-item (gensym "coll-item__")]
    `(do (put! ~buffer (count ~data) ::short ~byte-position)
         (run! (fn [~coll-item]
                 ~(serialize-1 sub-schema config wbuffer coll-item))
               ~data))))

(defn deserialize-coll [schema config {:keys [buffer byte-position] :as wbuffer}]
  (let [[coll-type _ sub-schema] (disj-composite schema)]
    `(->> (take! ~buffer ::short ~byte-position)
          (repeatedly (fn [] ~(deserialize-1 sub-schema config wbuffer)))
          (doall)
          ~(case coll-type
             ::vector vec
             ::list seq
             ::set set))))


(defn serialize-map [schema config {:keys [buffer byte-position] :as wbuffer} data]
  (let [[_ _ [key-schema value-schema]] (disj-composite schema)
        key   (gensym "key_")
        value (gensym "value_")]
    `(do (put! ~buffer (count ~data) ::short ~byte-position)
         (run! (fn [[~key ~value]]
                 ~(serialize-1 key-schema config wbuffer key)
                 ~(serialize-1 value-schema config wbuffer value))
               ~data))))

(defn deserialize-map [schema config {:keys [buffer byte-position] :as wbuffer}]
  (let [[_ _ [key-schema value-schema]] (disj-composite schema)]
    `(->> (take! ~buffer ::short ~byte-position)
          (repeatedly (fn [] [~(deserialize-1 key-schema config wbuffer)
                              ~(deserialize-1 value-schema config wbuffer)]))
          (doall)
          (into {}))))


(defn serialize-tuple [schema config wbuffer data]
  (let [[_ _ sub-schemas] (disj-composite schema)
        symbols  (repeatedly (count sub-schemas) #(gensym "tup-item_"))
        let-body (mapcat (fn [symbol index] [symbol `(get ~data ~index)])
                         symbols
                         (range))]
    `(let [~@let-body]
       ~@(->> (map vector symbols sub-schemas)
              (map (fn [[symbol sub-schema]]
                     (serialize-1 sub-schema config wbuffer symbol)))
              (doall)))))


(defn deserialize-tuple [schema config wbuffer]
  (let [[_ _ sub-schemas] (disj-composite schema)]
    `(vector ~@(doall (map (fn [sub-schema] (deserialize-1 sub-schema config wbuffer))
                           sub-schemas)))))


(defn serialize-record [schema config wbuffer data]
  (let [[_ _ args] (disj-composite schema)
        arg-pairs      (partition 2 args)
        symbols        (repeatedly (count arg-pairs) #(gensym "rec-item_"))
        let-body       (mapcat (fn [symbol [key _]] [symbol `(get ~data ~key)])
                               symbols
                               arg-pairs)
        symbol-schemas (map (fn [symbol [_ schema]] [symbol schema])
                            symbols
                            arg-pairs)]
    `(let [~@let-body]
       ~@(doall (map (fn [[symbol value-schema]] (serialize-1 value-schema config wbuffer symbol))
                     symbol-schemas)))))

(defn deserialize-record [schema config wbuffer]
  (let [[_ _ args] (disj-composite schema)]
    `(hash-map ~@(->> args
                      (partition 2)
                      (mapcat (fn [[key value-schema]]
                                [key (deserialize-1 value-schema config wbuffer)]))
                      (doall)))))


(defn serialize-sub-schema [schema config wbuffer data]
  (serialize-1 (get-in config [:schemas schema]) config wbuffer data))

(defn deserialize-sub-schema [schema config wbuffer]
  (deserialize-1 (get-in config [:schemas schema]) config wbuffer))


(defn processor [schema {:keys [schema-map]}]
  (println schema schema-map)
  (cond
    (primitive? schema) [serialize-primitive deserialize-primitive]
    (= ::string schema) [serialize-string deserialize-string]
    (= ::keyword schema) [serialize-keyword deserialize-keyword]
    (composite? schema) (case (first schema)
                          ::list [serialize-coll deserialize-coll]
                          ::vector [serialize-coll deserialize-coll]
                          ::set [serialize-coll deserialize-coll]
                          ::map [serialize-map deserialize-map]
                          ::tuple [serialize-tuple deserialize-tuple]
                          ::record [serialize-record deserialize-record])
    (schema-map schema) [serialize-sub-schema deserialize-sub-schema]
    :else [nil nil]))

(defn serialize-1 [schema config buffer data]
  (if-let [f (first (processor schema config))]
    (f schema config buffer data)))

(defn deserialize-1 [schema config buffer]
  (if-let [f (second (processor schema config))]
    (f schema config buffer)))


(defn serializer [schema {:keys [wbuffer] :as config}]
  (let [data (gensym "data_")]
    `(fn [~data]
       ~(serialize-1 schema config wbuffer data))))

(defn deserializer [schema config]
  (let [wbuffer (gensym "wbuffer_")]
    `(fn [~wbuffer]
       ~(deserialize-1 schema config wbuffer))))

;;;; ============================================================================================
;;;; Public API

(defn defconfig [name schemas & args]
  (let [args-map      (->> args (partition 2) (into {}))
        wbuffer       (make-wbuffer 10000 10000)
        schema-map    (bimap (keys schemas))
        keyword-map   (bimap (get args-map :keywords []))
        config        {:wbuffer     wbuffer
                       :schema-map  schema-map
                       :keyword-map keyword-map}
        serializers   (into {} (for [schema (keys schemas)]
                                 [schema `(serializer ~schema ~config)]))
        deserializers (into {} (for [schema (keys schemas)]
                                 [schema `(deserializer ~schema ~config)]))]
    (def name (assoc config :serializers serializers
                            :deserializers deserializers))))

(defmacro defconfigm [name schemas & args]
  `(let [name#          ~name
         schemas#       ~schemas
         args#          ~args
         args-map#      (->> args# (partition 2) (into {}))
         wbuffer#       (make-wbuffer 10000 10000)
         schema-map#    (bimap (keys schemas#))
         keyword-map#   (bimap (get args-map# :keywords []))
         config#        {:wbuffer     wbuffer#
                         :schema-map  schema-map#
                         :keyword-map keyword-map#}
         serializers#   (into {} (for [schema# (keys schemas#)]
                                   [schema# (serializer schema# config#)]))
         deserializers# (into {} (for [schema# (keys schemas#)]
                                   [schema# (deserializer schema# config#)]))]
     (def name# (assoc config# :serializers serializers#
                               :deserializers deserializers#))))

(mm '(defconfigm cat {::a ::int}))

(defconfig cat {::a ::int})
cat

(defn serialize
  [data schema {:keys [serializers wbuffer]}]
  (when-let [f (get serializers schema)]
    (reset-wbuffer! wbuffer)
    (f data)
    (unwrap-wbuffer wbuffer)))

(defn deserialize
  [bytes {:keys [deserializers schema-map]}]
  (let [[wbuffer schema-code] (wrap-bytes bytes)]
    (when-let [f (get deserializers (get schema-map schema-code))]
      (f wbuffer))))

;;;; ============================================================================================
;;;; Playground

; TODO second parameter map: ???, accept seqs as well as 'varargs', long as primitive
(defmacro foo [schema config]
  (let [buffer (gensym "buffer_")]
    `(fn [~buffer]
       ~(deserialize-1 schema config buffer))))

(comment
  (mm '(serializer [::record {}
                    :a ::int
                    :b ::string
                    :c [::tuple ::float ::int]]
                   {:wbuffer {:byte-position (atom 0)
                              :bit-position  (atom 0)
                              :buffer        :buffer}})))

(defn long->bytes [^long x]
  (for [i (range 7 -1 -1)]
    (-> x (bit-shift-right (* 8 i)) (bit-and 0xFF))))

(defn bytes->long [bs]
  (reduce (fn [l b]
            (bit-or b (bit-shift-left l 8)))
          0
          bs))
