(ns seria.half_dead
  #?(:clj
     (:import [java.nio ByteBuffer]))
  (:require [criterium.core :as c]))

(def mm #(clojure.pprint/pprint (macroexpand %)))
(def cc #(c/with-progress-reporting (c/quick-bench %)))

(set! *warn-on-reflection* true)

;;;; ============================================================================================
;;;; Util

(defn bimap [items]
  (->> items
       (map-indexed #(vector (- %1 32768) %2))
       (mapcat (fn [[a b]] [[a b] [b a]]))
       (into {})))

(defn primitive-size [type]
  (case type
    ::byte 1
    ::short 2
    ::int 4
    ::float 4
    ::double 8
    ::char 2
    ::boolean 1))

(defn short->bytes [x]
  (#?(:clj  (partial into-array Byte/TYPE)
      :cljs clj->js)
    [(-> x (bit-shift-right 8) (bit-and 0xFF) (unchecked-byte))
     (-> x (bit-and 0xFF) (unchecked-byte))]))

;;;; ============================================================================================
;;;; Buffers

(defprotocol AutoByteBuffer
  (take! [this type position])
  (put! [this data type position]))

(do #?(:clj  (extend-type ByteBuffer
               AutoByteBuffer
               (take! [this type position]
                 (let [p @position]
                   (swap! p + (primitive-size type))
                   (case type
                     ::byte (.get this ^int p)
                     ::short (.getShort this p)
                     ::int (.getInt this p)
                     ::float (.getFloat this p)
                     ::double (.getDouble this p)
                     ::char (.getChar this p)
                     ::boolean (-> this
                                   (.get ^int (quot p 8))
                                   (bit-test (rem p 8)))
                     nil)))
               (put! [this data type position]
                 (let [p @position]
                   (swap! position + (primitive-size type))
                   (case type
                     ::byte (.put this p (byte data))
                     ::short (.putShort this p (short data))
                     ::int (.putInt this p (int data))
                     ::float (.putFloat this p (float data))
                     ::double (.putDouble this p (double data))
                     ::char (.putChar this p (char data))
                     ::boolean (.put this (-> this
                                              (.get ^int (quot p 8))
                                              (bit-set (rem p 8))
                                              (unchecked-byte)))
                     nil))))

       :cljs (extend-type js/DataView
               AutoByteBuffer
               (take! [this type position]
                 (let [p @position]
                   (swap! position (+ p (primitive-size type)))
                   (case type
                     ::byte (.getInt8 this p)
                     ::short (.getInt16 this p)
                     ::int (.getInt32 this p)
                     ::float (.getFloat32 this p)
                     ::double (.getFloat64 this p)
                     ::char (.getUint8 this p)
                     ::boolean (-> this
                                   (.getInt8 (quot p 8))
                                   (bit-test (rem p 8)))
                     nil)))
               (put! [this data type position]
                 (let [p @position]
                   (swap! position (+ p (primitive-size type)))
                   (case type
                     ::byte (.setInt8 this p data)
                     ::short (.setInt16 this p data)
                     ::int (.setInt32 this p data)
                     ::float (.setFloat32 this p data)
                     ::double (.setFloat64 this p data)
                     ::char (.setUint8 this p data)
                     ::boolean (.setInt8 this (-> this
                                                  (.getInt8 (quot p 8))
                                                  (bit-set (rem p 8))
                                                  (unchecked-byte)))
                     nil))))))

;;;; ============================================================================================
;;;; Writing

(defn reset-positions! [{:keys [max-bits] :as config}]
  (assoc config :bit-position 16
                :byte-position (+ 2 max-bits)))

(defn make-buffer [size]
  #?(:clj  (ByteBuffer/allocate size)
     :cljs (js/DataView. (js/ArrayBuffer. size))))

(defn unwrap-buffer [{:keys [bit-position byte-position max-bits]} buffer]
  (let [bits-length  (int (Math/ceil (/ (- @bit-position 16) 8)))
        bytes-length (- @byte-position max-bits 2)
        total-length (+ 4 bits-length bytes-length)]
    #?(:clj  (let [buffer-bytes (.array ^ByteBuffer buffer)
                   bytes        (byte-array total-length)]
               (System/arraycopy (short->bytes bits-length) 0 buffer-bytes
                                 2 2)
               (System/arraycopy buffer-bytes 0 bytes
                                 4 bits-length)
               (System/arraycopy buffer-bytes max-bits bytes
                                 (+ 4 bits-length) bytes-length)
               bytes)
       :cljs (let [array-buffer (.-buffer buffer)
                   byte-view    (js/Int8Array. total-length)]
               (.set byte-view (short->bytes bits-length)
                     2)
               (.set byte-view (js/Int8Array. (.slice array-buffer 0 bits-length))
                     4)
               (.set byte-view (js/Int8Array. (.slice array-buffer max-bits (+ max-bits bytes-length)))
                     (+ 4 bits-length))
               (.-buffer byte-view)))))

;;;; ============================================================================================
;;;; Reading

(defn wrap-bytes [bytes]
  (let [buffer #?(:clj (ByteBuffer/wrap bytes)
                  :cljs (js/DataView. bytes))]
    [(take! buffer ::short (atom 0))
     (take! buffer ::short (atom 2))
     buffer]))

;;;; ============================================================================================
;;;; De/serialization

(def primitives
  #{::byte ::short ::int ::float ::double ::char})

(defn primitive? [type]
  (primitives type))

(def composites
  #{::list ::vector ::set ::map ::tuple ::record})

(defn composite? [schema]
  (and (vector? schema)
       (composites (first schema))))

(defn disj-composite [[a b & more]]
  (let [u  a
        v  (if (map? b) b {})
        w  (vec (if (or (map? b) (nil? b))
                  more
                  (cons b more)))
        w1 (first w)]
    (if (and (sequential? w1)
             (not (composite? w1)))
      [u v w1]
      [u v w])))


(declare serialize-any)
(declare deserialize-any)


(defn serialize-primitive [schema {:keys [bit-position byte-position]} buffer data]
  (let [position (if (= ::boolean schema) bit-position byte-position)]
    `(put! ~buffer ~data ~schema ~position)))

(defn deserialize-primitive [schema {:keys [bit-position byte-position]} buffer]
  (let [position (if (= ::boolean schema) bit-position byte-position)]
    `(take! ~buffer ~schema ~position)))


(defn serialize-string [_ {:keys [byte-position]} buffer data]
  (let [char (gensym "char_")]
    `(do (put! ~buffer (count ~data) ::short ~byte-position)
         (run! (fn [~char] (put! ~buffer ~char ::char ~byte-position))
               ~data))))

(defn deserialize-string [_ {:keys [byte-position]} buffer]
  `(->> (take! ~buffer ::short ~byte-position)
        (repeatedly (fn [] (take! ~buffer ::char ~byte-position)))
        (doall)
        (apply str)))


(defn serialize-keyword [_ {:keys [keyword-map byte-position]} buffer data]
  `(put! ~buffer (get ~keyword-map ~data) ::short ~byte-position))

(defn deserialize-keyword [_ {:keys [keyword-map byte-position]} buffer]
  `(get ~keyword-map (take! ~buffer ::short ~byte-position)))


(defn serialize-coll [schema {:keys [byte-position] :as config} buffer data]
  (let [[_ _ [sub-schema]] (disj-composite schema)
        coll-item (gensym "coll-item__")]
    `(do (put! ~buffer (count ~data) ::short ~byte-position)
         (run! (fn [~coll-item]
                 ~(serialize-any sub-schema config buffer coll-item))
               ~data))))

(defn deserialize-coll [schema {:keys [byte-position] :as config} buffer]
  (let [[coll-type _ [sub-schema]] (disj-composite schema)]
    `(->> (take! ~buffer ::short ~byte-position)
          (repeatedly (fn [] ~(deserialize-any sub-schema config buffer)))
          (doall)
          ~(case coll-type
             ::vector vec
             ::list seq
             ::set set))))


(defn serialize-map [schema {:keys [byte-position] :as config} buffer data]
  (let [[_ _ [key-schema value-schema]] (disj-composite schema)
        key   (gensym "key_")
        value (gensym "value_")]
    `(do (put! ~buffer (count ~data) ::short ~byte-position)
         (run! (fn [[~key ~value]]
                 ~(serialize-any key-schema config buffer key)
                 ~(serialize-any value-schema config buffer value))
               ~data))))

(defn deserialize-map [schema {:keys [byte-position] :as config} buffer]
  (let [[_ _ [key-schema value-schema]] (disj-composite schema)]
    `(->> (take! ~buffer ::short ~byte-position)
          (repeatedly (fn [] [~(deserialize-any key-schema config buffer)
                              ~(deserialize-any value-schema config buffer)]))
          (doall)
          (into {}))))


(defn serialize-tuple [schema config buffer data]
  (let [[_ _ sub-schemas] (disj-composite schema)
        symbols  (repeatedly (count sub-schemas) #(gensym "tup-item_"))
        let-body (mapcat (fn [symbol index] [symbol `(get ~data ~index)])
                         symbols
                         (range))]
    `(let [~@let-body]
       ~@(->> (map vector symbols sub-schemas)
              (map (fn [[symbol sub-schema]]
                     (serialize-any sub-schema config buffer symbol)))
              (doall)))))

(defn deserialize-tuple [schema config buffer]
  (let [[_ _ sub-schemas] (disj-composite schema)]
    `(vector ~@(doall (map (fn [sub-schema] (deserialize-any sub-schema config buffer))
                           sub-schemas)))))


(defn serialize-record [schema config buffer data]
  (let [[_ _ args] (disj-composite schema)
        arg-pairs      (partition 2 args)
        symbols        (repeatedly (count arg-pairs) #(gensym "rec-item_"))
        let-body       (mapcat (fn [symbol [key _]] [symbol `(get ~data ~key)])
                               symbols
                               arg-pairs)
        symbol-schemas (map (fn [symbol [_ schema]] [symbol schema])
                            symbols
                            arg-pairs)]
    `(let [~@let-body]
       ~@(doall (map (fn [[symbol value-schema]] (serialize-any value-schema config buffer symbol))
                     symbol-schemas)))))

(defn deserialize-record [schema config buffer]
  (let [[_ _ args] (disj-composite schema)]
    `(hash-map ~@(->> args
                      (partition 2)
                      (mapcat (fn [[key value-schema]]
                                [key (deserialize-any value-schema config buffer)]))
                      (doall)))))


(defn serialize-sub-schema [schema config buffer data]
  (serialize-any (get-in config [:schemas schema]) config buffer data))

(defn deserialize-sub-schema [schema config buffer]
  (deserialize-any (get-in config [:schemas schema]) config buffer))


(defn get-processor [schema {:keys [schema-map]}]
  (cond
    (primitive? schema) [serialize-primitive deserialize-primitive]
    (= ::string schema) [serialize-string deserialize-string]
    (= ::keyword schema) [serialize-keyword deserialize-keyword]
    (composite? schema) (case (first schema)
                          ::list [serialize-coll deserialize-coll]
                          ::vector [serialize-coll deserialize-coll]
                          ::set [serialize-coll deserialize-coll]
                          ::map [serialize-map deserialize-map]
                          ::tuple [serialize-tuple deserialize-tuple]
                          ::record [serialize-record deserialize-record])
    (schema-map schema) [serialize-sub-schema deserialize-sub-schema]
    :else [nil nil]))

(defn serialize-any [schema config buffer data]
  (when-let [f (first (get-processor schema config))]
    (f schema config buffer data)))

(defn deserialize-any [schema config buffer]
  (when-let [f (second (get-processor schema config))]
    (f schema config buffer)))


(defmacro make-serializer [schema config]
  (let [data   (gensym "data_")
        buffer (gensym "buffer_")]
    `(fn [~data ~buffer]
       ~(serialize-any schema config buffer data))))

(defmacro make-deserializer [schema config]
  (let [buffer (gensym "buffer_")]
    `(fn [~buffer]
       ~(deserialize-any schema config buffer))))

;;;; ============================================================================================
;;;; Public API

(defmacro defconfig [name schemas & args]
  (let [schemas       (eval schemas)
        args          (eval args)
        args-map      (->> args (partition 2) (into {}))
        wbuffer       (make-buffer 10000 10000)
        schema-map    (bimap (keys schemas))
        keyword-map   (bimap (get args-map :keywords []))
        config        {:schemas     schemas
                       :wbuffer     wbuffer
                       :schema-map  schema-map
                       :keyword-map keyword-map}
        serializers   (into {} (for [schema (keys schemas)]
                                 [schema (make-serializer schema config)]))
        deserializers (into {} (for [schema (keys schemas)]
                                 [schema (make-deserializer schema config)]))]
    `(def ~name ~(assoc config :serializers serializers
                               :deserializers deserializers))))

(defn serialize
  [data schema {:keys [serializers buffer schema-map]}]
  (when-let [f (get serializers schema)]
    (put! buffer (schema-map schema) ::short (atom 0))
    (reset-positions! buffer)
    (f data buffer)
    ;(unwrap-buffer buffer)
    ))

(defn deserialize
  [bytes {:keys [deserializers schema-map]}]
  (let [[schema-code buffer bit-length] (wrap-bytes bytes)]
    (when-let [f (get deserializers (get schema-map schema-code))]
      (f buffer))))

;;;; ============================================================================================
;;;; Toplevel testing

(defconfig test_1 {::a [::vector (identity ::b)]
                   ::b [::record {}
                        :foo [::tuple ::int ::int]
                        :bar [::record {}
                              :qux ::int]]})

(serialize {:foo [4 6]
            :bar {:qux 30}}
           ::b
           my-config125)

;;;; ============================================================================================
;;;; De/serialization testing

(defmacro make-serializer-m [schema config]
  (let [data   (gensym "data_")
        buffer (gensym "buffer_")]
    `(fn [~data ~buffer]
       ~(serialize-any schema config buffer data)
       nil)))

(defmacro make-deserializer-m [schema config]
  (let [buffer (gensym "buffer_")]
    `(fn [~buffer]
       ~(deserialize-any schema config buffer)
       nil)))