(ns weir.core
  "Macros for the weir library.")

(defmacro defevent
  "Defines an event listener. It requires a namespace qualified `name`,
  symbol, identifying the kind of event it listens to.

  The `strategy` argument is a keyword, which gives a hint what kind
  of strategy should be followed, when the number of events in the log
  is getting too large and needs a clean-up/compaction. Currently it
  has two options:

  * `:keep-all` - do not clean up or compact events of this kind if they
  are not read yet.

  * `:keep-latest` - it is allowed to compact this kind of event, such that
  at least the latest unread event is kept.

  Next is a binding vector, just as with a normal function. It acts as
  a 3-arity function. It receives the event (a namespaced keyword),
  the data for that event, and a context map. This context map
  contains the following:

  * `:conn` - the DataScript connection for quering and transacting.

  An example:

  ```
  (defevent my.app/my-event :all
    [event data {:keys [conn emit-fn] :as context}]
    ...)
  ```

  These event handlers are only suitable for non side effecting logic,
  such as updating the local storage. Use `defevent!` for side
  effects."
  [name strategy bindings & body]
  (let [msg-key (case strategy
                  :keep-all    nil
                  :keep-latest `(quote ~name))]
    `(do
       (defmethod weir.core/event-key ~(keyword name)
         [event#]
         ~msg-key)
       (defmethod weir.core/event-handler ~(keyword name)
         ~bindings
         ~@body))))

(defmacro defevent!
  "Same as `defevent`, but for side effects, as these are only
  executed when weir is \"live\", i.e. not reloading. This handler is
  run _after_ the `defevent` handler of the same event.

  The context map (the third argument of the handler) now contains
  extra entries:

  * `:emit-fn` - a 1-to-2-arity function to emit a new event onto the
  franz topic, where the first argument is a namespaced keyword
  identifying the event kind, and the optional second argument is the
  event data.

  * `:send-fn` - The send function of sente."
  [name strategy bindings & body]
  (let [msg-key (case strategy
                  :keep-all    nil
                  :keep-latest `(quote ~name))]
    `(do
       (defmethod weir.core/event-key ~(keyword name)
         [event#]
         ~msg-key)
       (defmethod weir.core/event-handler! ~(keyword name)
         ~bindings
         ~@body))))
