(ns weir.core
  "The core API."
  (:require-macros [reagent.ratom :as rm])
  (:require [datascript.core :as d]
            [franz.core :as f]
            [franz.protocol :as fp]
            [reagent.core :as r]
            [taoensso.sente :as s]))

;;; Multimethods for internals.

(defmulti event-key
  (fn [event] event))

(defmethod event-key :default
  [_]
  nil)

(defmulti event-handler
  (fn [data context]
    (:event context)))

(defmethod event-handler :default
  [_ _]
  ::unhandled)

(defmulti event-handler!
  (fn [data context]
    (:event context)))

(defmethod event-handler! :default
  [_ _]
  ::unhandled)


;;; Event handling

(defonce weir (atom nil))

(defmethod event-handler :caughtup
  [_ _ _]
  (swap! weir assoc :live? true))

(defn- watch-franz
  [log-fn context]
  (f/full-handler
   (fn [offset [key [event data]]]
     (if event
       (let [context (assoc context :event event)
             live?   (-> weir deref :live?)
             result  (try
                       (event-handler data (select-keys context [:conn :event]))
                       (catch :default e
                         (log-fn "Error while handling" event ":" e)))
             _       (when (not= result ::unhandled)
                       (log-fn "Handled event at" offset ":" event))
             result! (when live?
                       (try
                         (event-handler! data context)
                         (catch :default e
                           (log-fn "Error while handling side effecting" event ":" e))))
             _       (when (and live? (not= result! ::unhandled))
                       (log-fn "Handled side effecting event at" offset ":" event))]
         (when (= [result result!] [::unhandled ::unhandled])
           (log-fn "Unhandled event at" offset ":" event)))
       (log-fn "Got nil message at" offset)))))

(defn- watch-db
  [app-db]
  (fn [_ _ _ new-db] (reset! app-db new-db)))

(defn- sente-router
  [log-fn topic]
  (fn [msg]
    (let [{:keys [id ?data]} msg]
      (if (= id :chsk/recv)
        (let [[event data] ?data]
          (f/send! topic (event-key event) [event data]))
        (log-fn "Got non-event sente message:" (pr-str msg))))))

(defn- event-emitter
  [log-fn topic]
  (fn emit-event
    ([event]
     (emit-event event nil))
    ([event data]
     (log-fn "Emitting event:" event)
     (f/send! topic (event-key event) [event data]))))


;;; Reloading and initialization.

(defn ^:export reload!
  "Reload by cleaning the database and replaying all events that do
  not perform IO with external systems."
  []
  (let [{:keys [router topic conn schema init-tx emit-fn log-fn sente app-db]} @weir]
    ;; Stop adding new events from the server to the franz queue.
    (when router (router))
    ;; Process any outstanding events on the queue.
    (f/flush! topic)
    ;; Stop updating app-db while replaying.
    (remove-watch conn :watch-db)
    ;; Unsubscribe from the event queue.
    (f/unsubscribe! topic :watch-franz)
    ;; Clear the database.
    (d/reset-conn! conn (d/empty-db schema))
    ;; Add the initial transactions to the cleaned database.
    (when init-tx (d/transact! conn init-tx))
    ;; Set the live flag to false.
    (swap! weir assoc :live? false)
    ;; Add an caught-up event at the end of the queue.
    (f/send! topic :caughtup [:caughtup])
    (fp/-compact! topic)
    ;; Subscribe again to the topic, starting from position 0, replaying all events
    (let [context {:conn    conn
                   :emit-fn emit-fn
                   :send-fn (:send-fn sente)}]
      (f/subscribe! topic :watch-franz (watch-franz log-fn context) {:blocking-clean true}))
    ;; Make sure all events have been replayed.
    (f/flush! topic)
    ;; Make sure that the app-db reactive atom always has the latest
    ;; database value.
    (reset! app-db (d/db conn))
    (add-watch conn :watch-db (watch-db app-db))
    ;; Start putting events from the server to the franz queue.
    (when sente
      (swap! weir assoc :router
             (->> (sente-router log-fn topic)
                  (s/start-client-chsk-router! (:ch-recv sente)))))))

(defn ^:export initialize!
  "This function sets everything in motion. It will set everything up
  and connect sente to your server. It takes no required arguments,
  though options can be specified using keyword arguments.

  It returns a map, with several entries, but the important two are
  `:app-db` and `:emit-fn`. The values of those can be passed to your
  view components, as soon as you mount them. For example:

  ```
  (defn ^:export mount []
    (let [weir (weir/initialize!)]
      (r/render-component [my-component (:app-db weir) (:emit-fn weir)]
                          (.getElementById js/document \"my-id\"))))
  ```

  The `initialize!` function can take the following keyword arguments:

  * `:sente-path` - The path the sente client should connect to on the
  server. Default is `\"/chsk\"`. Set this to `nil` to skip starting
  sente at all.

  * `:sente-opts` - The options for the sente client. Default is
  `{:type :auto}`.

  * `:schema` - The DataScript schema. Default is `{}`.

  * `:init-tx` - An initial DataScript transaction. Default is
  `nil` (no transaction).

  * `:topic-opts` - The options passed when creating the franz topic.
  This is a more advanced feature; the defaults should work quite
  well.

  * `:log-fn` - A variable arity function, taking objects forming a
  debug message. Expects the objects to be interposed with a space.
  Default is `println`."
  [& {:keys [sente-path sente-opts schema init-tx init-events log-fn topic-opts]
      :or   {sente-path "/chsk"
             sente-opts {:type :auto}
             log-fn     println}}]
  (if @weir
    (throw (ex-info "Call to initialize! while already initialized" {}))
    (let [;; Creating the datascript database connection.
          conn    (d/create-conn schema)
          ;; Create a reactive atom, which will watch the database.
          app-db  (r/atom (d/db conn))
          ;; Create a franz topic for the events.
          topic   (f/topic nil topic-opts)
          ;; Create a function that puts events on the franz topic.
          emit-fn (event-emitter log-fn topic)
          ;; Start sente, when a path has been supplied.
          sente   (when sente-path
                    (s/make-channel-socket! sente-path sente-opts))]
      ;; Add the initial events to the franz topic.
      (doseq [[event data] init-events]
        (emit-fn event data))
      ;; Store all the configuration and stateful parts.
      (reset! weir {:schema  schema
                    :init-tx init-tx
                    :log-fn  log-fn
                    :conn    conn
                    :app-db  app-db
                    :topic   topic
                    :emit-fn emit-fn
                    :sente   sente
                    :live?   false})
      ;; Perform a (re)load to set everything in motion.
      (reload!)
      ;; Return the context for the reagent view.
      {:app-db  app-db
       :emit-fn emit-fn})))

(defn ^:export stop!
  "Clear the database and queue, and bring down any listener and
  connection from weir. Please unmount your view as well, as the
  `app-db` and `emit-fn` are invalidated as well."
  [])
