(ns puget.data
  "Code to handle custom data represented as tagged EDN values."
  (:import
    (java.net URI)
    (java.util Date TimeZone UUID)))


(defprotocol ExtendedNotation
  "Protocol for types which use extended notation for EDN representation."

  (->edn
    [value]
    "Converts the given value into a tagged value representation for EDN
    serialization. Returns a `TaggedLiteral` record."))


; TODO: remove this when CLJ-1424 merges
; http://dev.clojure.org/jira/browse/CLJ-1424
(defrecord TaggedLiteral
  [tag form]

  ExtendedNotation

  (->edn
    [this]
    this)


  Object

  (toString
    [this]
    (str \# tag \space (pr-str form))))


;; Remove automatic constructor functions.
(ns-unmap *ns* '->TaggedLiteral)
(ns-unmap *ns* 'map->TaggedLiteral)


(defmethod print-method TaggedLiteral
  [v ^java.io.Writer w]
  (.write w (str v)))


(defn tagged-literal
  "Creates a generic tagged value record to represent some EDN value. This is
  suitable for use as a default-data-reader function."
  [tag value]
  {:pre [(symbol? tag)]}
  (TaggedLiteral. tag value))


(defn tagged-literal?
  "Returns true if the given value is a tagged-literal form."
  [value]
  (instance? TaggedLiteral value))



;; ## Extension Functions

(defmacro extend-tagged-value
  "Defines an EDN representation for a type `t`. The tag will be the symbol
  given for `tag` and the literal form will be generated by applying `expr` to
  the original value."
  [t tag expr]
  `(let [value-fn# ~expr]
     (extend-type ~t
       ExtendedNotation
       (->edn
         [this#]
         (tagged-literal ~tag (value-fn# this#))))))


(defmacro extend-tagged-str
  "Defines an EDN representation for the given type by converting it to a
  string form."
  [t tag]
  `(extend-tagged-value ~t ~tag str))


(defmacro extend-tagged-map
  "Defines an EDN representation for the given type by converting it to a
  map form."
  [t tag]
  `(extend-tagged-value ~t ~tag
     (comp (partial into {}) seq)))



;; ## Standard EDN Types

(defn- format-utc
  "Produces an ISO-8601 formatted date-time string from the given Date."
  [^Date date]
  (-> "yyyy-MM-dd'T'HH:mm:ss.SSS-00:00"
      java.text.SimpleDateFormat.
      (doto (.setTimeZone (TimeZone/getTimeZone "GMT")))
      (.format date)))


;; `inst` tags a date-time instant represented as an ISO-8601 string.
(extend-tagged-value Date 'inst format-utc)


;; `uuid` tags a universally-unique identifier string.
(extend-tagged-str UUID 'uuid)
