(ns lonocloud.graph
  (:require [n01se.deltype :refer [deftype]])
  (:refer-clojure :exclude [deftype assoc-in]))

;; Generic code goes here
(defn assoc-in
  "Regular assoc-in but with (correct) handling of the degenerate case of an
  empty path in which coll is entirely replaced with value."
  [coll path value]
  (if (empty? path)
    value
    (clojure.core/assoc-in coll path value)))

;; Define NodeMaker and fns
(defprotocol NodeMaker
  (node- [_ graph focus visited] "Return a node based on self."))

(defn node
  "Return a node using graph."
  [graph & [focus visited]]
  (node- (get-in graph focus) graph (vec focus) (set #{})))

;; Define Node and fns
(defprotocol Node
  (graph [_] "Return the current graph.")
  (focus [_] "Return the current focus."))

(defn value
  "Return the collection that node wraps."
  [n]
  (get-in (graph n) (focus n)))

(defn refocus
  "Return a node with focus into old-node's graph."
  [old-node & focus]
  (node (graph old-node) focus))

;; Define Link type and fns
(deftype Link [focus]
  Object
  (toString [_] (str focus))
  NodeMaker
  (node- [_ graph _ visited]
    (when-not (contains? visited focus)
      (node graph focus (conj visited focus)))))

(defn link
  "Construct a link with focus specified as a vararg."
  [& focus]
  (Link. (vec focus)))

;; Define NodeMap and NodeVector types
(deftype NodeMap [m g f] ; [m]ap, [g]raph, [f]ocus
  :delegate [m n01se.deltype.IMap]

  NodeMaker
  (node- [_ graph focus _]
    (NodeMap. m graph focus))

  Node
  (graph [_] (assoc-in g f m))
  (focus [_] f)

  n01se.deltype.IMap

  ;; methods dealing with digging into a graph.
  (invoke [self k] (.valAt self k))
  (invoke [self k d] (.valAt self k d))
  (entryAt [self k]
    (when (contains? m k)
      [k (.valAt self k)]))
  (valAt [self k] (.valAt self k nil))
  (valAt [self k d]
    (node (assoc-in g f (if (contains? m k)
                          m
                          (assoc m k d)))
          (conj f k)))

  ;; methods dealing with updating the graph.
  (cons [self [k v]] (.assoc self k v))
  (assocEx [self k v] (.assoc self k v))
  (assoc [_ k v]
    (if (extends? Node (class v))
      (node (graph v) f)
      (NodeMap. (.assoc m k v) g f))))

(deftype NodeVector [v g f] ; [v]ector, [g]raph, [f]ocus
  :delegate [v n01se.deltype.IVector]

  NodeMaker
  (node- [_ graph focus _]
    (NodeVector. v graph focus))

  Node
  (graph [_] (assoc-in g f v))
  (focus [_] f)

  n01se.deltype.IVector

  ;; methods dealing with digging into a graph.
  (invoke [self i] (.valAt self i))
  (nth [self i] (.valAt self i))
  (valAt [self i]
    (node (assoc-in g f v)
          (conj f i)))

  ;; methods dealing with updating the graph.
  (assocN [self i v] (.assoc self i v))
  (assoc [_ i v]
    (if (extends? Node (class v))
      (node (graph v) f)
      (NodeVector. (.assoc v i v) g f))))

;; Extend NodeMaker over built in classes.
(extend-protocol NodeMaker
  ;; normal objects just return themselves.
  Object
  (node- [v _ _ _] v)

  nil
  (node- [v _ _ _] v)

  ;; while collections wrap themselves in a Node* type.
  clojure.lang.APersistentMap
  (node- [m graph focus _] (NodeMap. m graph focus))

  clojure.lang.APersistentVector
  (node- [v graph focus _] (NodeVector. v graph focus)))

(def demo
  (node {:log {
               :items []}
         :datomic {
                   :log (link :log)
                   :url "datomic://mem"
                   :key "xxyyzz"
                   :conn nil}
         :warg {
                :lg (link :log)
                :db (link :datomic)
                :deciders []
                :workers []}
         :cmdb {
                :stuff nil}
         :add-ntd-wf {
                      :log (link :log)
                      :dbg-state {
                                  :sub-state (link :warg :workers)}
                      :cmdb (link :cmdb)} }))

