(ns deftype+
  "Augmented deftype sporting a new :delegate option."
  :require [lonocloud.synthread :as ->])

(def letters
  (map (comp symbol str)
       "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"))

(defn map-items
  "Apply f to all key-value pairs in m. f should take two arguments (a key and
  its associated value) and return a sequence of zero or two values (the key and
  value).  If the returned sequence has zero values, no key/value is inserted
  into the return map."
  [f m]
  (reduce-kv
    (fn [m k v]
      (let [result (f k v)]
        (if (empty? result)
          m
          (assoc m (first result) (second result)))))
    (empty m)
    m))

(defn map-keys
  "Apply f to all keys in m."
  [f m]
  (map-items (fn [k v] [(f k) v]) m))

(defn map-vals
  "Apply f to all values in m."
  [f m]
  (map-items (fn [k v] [k (f v)]) m))

(defmulti monoid?
  "Return true if named method is a monoid"
  identity)

(defmethod monoid? :default
  [obj]
  (println "Default Monoid? No!" )
  false)

(defn interface? [klass]
  (.isInterface klass))

(defn get-ifaces [obj]
  (if (class? obj)
    (if (interface? obj)
      (cons obj (supers obj))
      (filter interface? (supers obj)))
    (when (map? obj) ;; assume it is a protocol
      [obj])))

(defn get-iname [obj]
  (if (class? obj)
    (symbol (.getName obj))
    (when (map? obj)
      (.sym (:var obj)))))

(defn get-sigs [obj]
  (->>
    (if (class? obj)
      (->>
        (.getDeclaredMethods obj)
        (map (juxt #(-> % .getName symbol)
                   #(-> % .getParameterTypes count (take letters) vec)))
        (reduce
          (fn [m [mname arglist]]
            (update-in m [mname :arglists] conj arglist))
          {}))
      (->>
        (:sigs obj)
        (map-items #(list (symbol (name %1))
                          (update-in %2 [:arglists]
                                     (partial (map (comp vec rest))))))))

    ;; associate monoid status with methods
    (map-vals #(if (contains? % :monoid)
                 %
                 (assoc % :monoid false)))))

(defn lookup [x]
  (let [x (resolve x)]
    (if (var? x)
      (deref x)
      x)))

;; use the hidden parsing functions in clojure.core
(def parse-opts (resolve 'clojure.core/parse-opts))
(def parse-impls (resolve 'clojure.core/parse-impls))
(defn parse-method-names [specs]
  (->> specs
    (filter list?)
    (map first)
    set))

(defn parse-delegate [delegates]
  (for [[field iname] (partition-all 2 delegates)]
    [field (map (juxt get-iname get-sigs)
                (get-ifaces (lookup iname)))]))

(defn update-impls [impls mnames delegates tname fields]
  (-> {:impls impls
       :mnames mnames}
    (->/for [[field ifaces] delegates
             [iname sigs] ifaces
             [mname {:keys [arglists monoid]}] sigs ]
      (->/as {:keys [mnames]}
        (->/when-not (contains? mnames mname
          (->/assoc :mnames (conj mname))
          (->/for [arglist arglists]
            (->/let [call `(~(symbol (str "." mname)) ~field ~arglist)
                     call (if monoid
                            `(~(symbol (str tname ".")) ~@(replace {field call}
                                                                   fields))
                            call)
                     method `(~mname [~self ~@arglist] ~call)]
              (update-in [:impls iname] conj method)))))))
    :impls))

;; supporting functions for deftype+ macro
(defn emit-delegation-methods
  "Emit a delegation method. If the method is an updater, be sure to wrap the
  results in a new instance of the overall type."
  [field fname arities monoid? tname fields]
  (let [fname-meth (symbol (str "." fname))
        tname-ctor (symbol (str tname "."))
        self (gensym (.toLowerCase (str tname "-")))]
    (for [arity arities]
      (let [fargs (repeatedly arity (partial gensym "arg-"))
            fbody `(~fname-meth ~field ~@fargs)
            fbody (if monoid?
                    `(~tname-ctor ~@(replace {field fbody} fields))
                    fbody)]
        `(~fname [~self ~@fargs] ~fbody)))))

#_(defn add-delegations
  "Work through delegation-pairs adding any missing delegation methods to
  topic."
  [topic delegation-pairs]
  (reduce
    (fn [topic [field dname]]
      (assert (some #(= field %) (:fields topic))
              (str "Unable to delegate to " field
                   " since it is not a declared field in "
                   (:tname topic) (:fields topic)))
      (reduce-kv
        (fn [topic mname {:keys [arities monoid?]}]
          (if (contains? (:mnames topic) mname)
            topic
            (-> topic
              (update-in [:impls dname]
                         (fnil concat [])
                         (emit-delegation-methods
                                   field
                                   mname
                                   arities
                                   monoid?
                                   (:tname topic)
                                   (:fields topic) ))
              (update-in [:mnames] conj mname)
              )))
        topic
        (get-methods (eval dname))))
    topic
    (partition-all 2 delegation-pairs)))

(defmacro deftype+
  "Built on top of clojure's standard deftype and is the same in every way
  except for the new :delegate option.

  The :delegate option takes as a value a vector of pairs of field names to
  class/interface/protocol names. Each field name must also appear in the type's
  overall field vector. A field name repeated and so associated with multiple
  class/interface/procotols.

  Example:
  (deftype+ MyMap [sub-map]
    :delegate [sub-map clojure.lang.PersistentHashMap] ;; act like a map

    clojure.lang.IPersistentMap
    ;; manually define just the assoc behavior
    (assoc [_ k v]
      (println \"Associng a new key into map: \" k)
      (MyMap. (.assoc sub-map k v))))

  ;; MyMap acts just like a map.
  (-> (MyMap. {:a 1})
    (assoc :b 2))"

  ;; 1. parse options ;=> {options}
  ;; 2. parse delegation option ;=> [[field, {iname [sigs]}]]
  ;; 3. parse impls ;=> [[iname [methods]]]
  ;; 4. parse mname set ;=> #{mnames}
  ;; 5. augment impls using ifaces
  ;; 6. append new impls using unused ifaces
  ;; 7. emit toplevel (deftype) form

  [tname fields & opts+specs]
  (let [[opts specs] (parse-opts opts+specs)
        delegates (parse-delegate (:delegate opts))
        mnames (parse-method-names specs)
        impls (parse-impls specs)
        impls (update-impls impls mnames delegates tname fields)
        specs (mapcat #(cons (first %) (second %)) impls)
        opts (mapcat identity (dissoc opts :delegate))]
    `(deftype ~tname ~fields ~@opts ~@specs)))

;; This is a library of delegate-specs. Each key in the spec is an interface or
;; protocol and each value is a submap with method specific directives. This
;; submap has method names as keys and one of three keywords as values: :get
;; (default) :update :skip. If no method is specified the submap :get is
;; assumed. :get means that the results of the delegated method are directly
;; returned. :update means that the results are wrapped in a new instance of the
;; overall type. :ignore will supply no delegate for this method.

(def ClojureMeta
  '{clojure.lang.IObj
      {withMeta :update}
    clojure.lang.IMeta
      {meta :get}})

(def ClojureCommon
  (merge ClojureMeta
         '{clojure.lang.IPersistentCollection
             {count :get
              cons :update
              empty :update}
           clojure.lang.Seqable
             {seq :get}
           clojure.lang.ILookup
             {valAt :get}
           ;; invoke has some kind of weird arity issue
           ;;clojure.lang.IFn
           ;;  {invoke :get}
           java.lang.Iterable
             {iterator :get}}))

(def ClojureMap
  (merge ClojureCommon
         '{clojure.lang.IPersistentMap
             {assoc :update
              assocEx :update
              without :update}
           clojure.lang.Associative
             {assoc :ignore
              containsKey :get
              entryAt :get} }))

(def ClojureSet
  (merge ClojureCommon
         '{clojure.lang.IPersistentSet
            {disjoin :update
             contains :get
             get :get}}))

