(ns n01se.logds)

(use 'clojure.repl)

;; my deftype

;;;; delegate option

;; code to get the methods of interfaces and protocols
(defmulti get-methods
  "Return a map of all method names to their arity."
  class)

(defmethod get-methods clojure.lang.PersistentArrayMap
  [protocol]
  (->> (:sigs protocol)
    vals
    (map (juxt :name
               #(-> % :arglists first count)))
    (into {})))

(defmethod get-methods java.lang.Class
  [interface]
  (->> interface
    (.getDeclaredMethods)
    (map (juxt #(-> % .getName symbol)
               #(-> % .getParameterTypes count)))
    (into {})))

;; use the hidden parsing functions in clojure.core
(def parse-opts (resolve 'clojure.core/parse-opts))
(def parse-impls (resolve 'clojure.core/parse-impls))

(defn emit-delegation-method
  "Emit a delegation method. If the method is a monoid, be sure to wrap the
  results in a new instance of the overall type."
  [delegate fname arity monoid? tname fields]
  (let [fname-meth (symbol (str "." fname))
        tname-ctor (symbol (str tname "."))
        fargs (repeat arity (gensym "arg-"))
        self (gensym (str (.toLowerCase tname) "-"))
        fbody `(~fname-meth ~delegate ~@fargs)
        fbody (if monoid?
                `(~tname-ctor ~@(replace {delegate fbody} fields))
                fbody)]
    `(~fname [~self ~@fargs] ~fbody)))

(defn mapify-methods
  "Convert the vector of methods into a map with the name of each method as the
  key."
  [impls]
  (reduce-kv
    (fn [impls iname methods]
      (assoc impls
             iname
             (reduce (fn [m [name :as method]]
                       (assoc m name method))
                     {}
                     methods)))
    impls
    impls))

(defn add-delegations
  "Work through the delegation-specs adding all needed delegation methods to
  impls."
  [impls delegation-specs tname fields]
  (reduce-kv
    (fn [impls delegate impl-specs]
      (reduce-kv
        (fn [impls iname monoids]
          (assoc impls iname
                 (reduce-kv
                   (fn [methods name arity]
                     (if (contains? methods name)
                       methods
                       (assoc methods name
                              (emit-delegation-method
                                delegate
                                name arity
                                (contains? monoids name)
                                tname
                                fields))))
                   (get impls iname {})
                   (get-methods (resolve iname)))))
        impls
        (eval impl-specs)))
    impls
    delegation-specs))

(defn flatten-maps
  "Flatten the impl map (and nested method map) into a vector."
  [impls]
  (reduce-kv
    (fn [specs iname methods]
      (concat specs
              [iname]
              (vals methods)))
    []
    impls))

(defmacro deftype+
  "Built on top of clojure's standard deftype. Takes a map of options and specs
  that is generated into a digestable form for deftype"
  [tname fields & opts+specs]
  (let [[opts specs] (parse-opts opts+specs)
        specs (-> specs
                parse-impls
                mapify-methods
                (add-delegations (:delegate opts) tname fields)
                flatten-maps)
        opts (-> opts
               (dissoc :delegate)
               (->> (mapcat identity)))]
    `(deftype ~tname ~fields ~@opts ~@specs)))

(comment "Example usage"
         (deftype Foo+ [m x]
           :delegate {x '{clojure.lang.IPersistentSet
                          #{disjoin}}}
           clojure.lang.IPersistentSet
           (contains [_ e] false)))
;; ongoing loggable work
(defprotocol Log
  (with-log [s l] "Return a logging version of s starting with l.")
  (add-log [s] "Return a logging version of s.")
  (log [s] "Return the change log associated with s."))

(defn disj-or-conj
  "disj or conj elem from or into s."
  [s elem]
  ((if (contains? s elem) disj conj) s elem))

(deftype LogSet [^clojure.lang.IPersistentSet s l]
  clojure.lang.IPersistentSet
  (contains [_ e] (contains? s e))
  (seq [_] (seq s))
  (count [_] (count s))

  (disjoin [original e]
    (if (contains? s e)
      (LogSet. (disj s e) (disj-or-conj l e))
      original))

  (cons [original e]
    (if (contains? s e)
      original
      (LogSet. (conj s e) (disj-or-conj l e))))

  clojure.lang.IFn
  (invoke [_ e] (s e))

  Log
  (with-log [_ l] (LogSet. s l))
  (add-log [_] (LogSet. s #{}))
  (log [_] l))

(extend-type clojure.lang.IPersistentSet
  Log
  (with-log [s l] (LogSet. s l))
  (add-log [s] (LogSet. s #{}))
  (log [_] #{}))

(defmacro defmap [name args & body]
  (let [name' (symbol (str name "."))]
    `(deftype ~name [m# ~@args]
       clojure.lang.IPersistentMap
       (~'assoc [_# k# v#]
         (~name' (.assoc m# k# v#)))
       )))

(comment
       (~'assocEx [_# k# v#]
         (~name' (.assocEx m# k# v#)))
       (~'without [_# k#]
         (~name' (.without m# k#)))
  )

(deftype Person [contents]
    clojure.lang.IPersistentMap
    (assoc  [_ k v]
          (Person.  (.assoc contents k v)))
    (assocEx  [_ k v]
          (Person.  (.assocEx contents k v)))
    (without  [_ k]
          (Person.  (.without contents k)))

    java.lang.Iterable
    (iterator  [this]
          (.iterator  (contents contents)))

    clojure.lang.Associative
    (containsKey  [_ k]
          (.containsKey  (contents contents) k))
    (entryAt  [_ k]
          (.entryAt  (contents contents) k))

    clojure.lang.IPersistentCollection
    (count  [_]
          (.count  (contents contents)))
    (cons  [_ o]
          (Person.  (.cons contents o)))
    (empty  [_]
          (.empty  (contents contents)))
    (equiv  [_ o]
          (and  (isa?  (class o) Person)
                        (.equiv  (contents contents)
                           (.(contents contents) o))))

    clojure.lang.Seqable
    (seq  [_]
          (.seq  (contents contents)))

    clojure.lang.ILookup
    (valAt  [_ k not-found]
          (.valAt  (contents contents) k not-found)))

