(ns n01se.ins
  "Initialization namespace support. Add ability to define a namespace and
  initialize that namespace with parameters. Defines the init-ns function and a
  pair of convenience macros: def-ns and req-ns.

  ;; Example using just init-ns
  (ns a)
  (defn init! [x]
    (defn foo [y] (+ x y)))

  (ns b)
  (init-ns 'a a/init! 1)
  (a/foo 1) ;=> 2

  ;; Similar example using def-ns and req-ns
  (def-ns a [x]
    (defn foo [y] (+ x y)))

  (ns b)
  (req-ns a :init [1])
  (a/foo 1) ;=> 2
  ")

;; API
(defn init-ns
  "Run (apply f args) within the namespace specified by name creating it if
  needed."
  [name f & args]
  (binding [*ns* (create-ns name)]
    (apply f args)))

(defmacro ns2 [n]
  `(~n))

(defmacro def-ns [name & body]
  `(binding [*ns* (create-ns ~name)]
     (defn- init-ns! ~@body)))

(defn req-ns [sym & {:as opts}]
  (let [{:keys [as init]} opts
        target (or as sym)
        req-opts (select-keys [:reload :reload-all])
        filter-opts (select-keys [:exclude :only :refer :rename])]
    (apply require sym req-opts)
    (init-ns target (-> (symbol (str sym) "init-ns!") var deref) init)
    (apply refer target filter-opts)))
