(ns navaratine.xlsx.shared-strings
  (:require [clojure.data.xml :as xml]
            [navaratine.xlsx.utils :as u]
            [com.rpl.specter :as sr]))


(defn decoder
  "Create a transducer that replaces sharedStrings.xml references with strings."
  [ref->str]
  (map #(sr/transform [u/each-cell (u/attr= :t "s") u/cell-text] ref->str %)))


(defn encoder
  "Create a transducer that replaces strings with sharedStrings.xml references."
  []
  (let [str->ref (volatile! {})
        encode   (fn [s]
                   (let [id (get @str->ref s (count @str->ref))]
                     (vswap! str->ref assoc s id)
                     (str id)))]
    [(map #(sr/transform [u/each-cell (u/attr= :t "s") u/cell-text] encode %))
     str->ref]))


(defn counter
  "Create a transducer that counts occurences of sharedStrings.xml references."
  []
  (let [cnt (volatile! 0)]
    [(map #(sr/transform
            [u/each-cell (u/attr= :t "s")]
            (fn [c] (vswap! cnt inc) c)
            %))
     cnt]))


(defn parse-shared-strings-xml
  "Parse xl/sharedStrings.xml and create ref->str map."
  [shared-strings-xml]
  (into {}
        (sr/select
         [(sr/subselect
           [(u/tag= :X/sst) u/children (u/tag= :X/si) u/children (u/tag= :X/t) u/text])
          sr/INDEXED-VALS
          (sr/view (fn [[i s]] [(str i) s]))]
         shared-strings-xml)))


(defn generate-shared-strings-xml
  "Generate xl/sharedStrings.xml based on mapping created by encoder and
   number of strings occuring in the generated sheets."
  [shared-strings-xml str->ref cnt]
  (let [ss-table
        (sequence
         (comp (map (fn [[s _]] [(u/resolve :X/si) {} [(u/resolve :X/t) {} s]]))
               (map xml/sexp-as-element))
         (sort-by second @str->ref))]
    (->> shared-strings-xml
         (sr/setval [(u/tag= :X/sst) :content] ss-table)
         (sr/setval [(u/tag= :X/sst) :attrs :count] (str @cnt))
         (sr/setval [(u/tag= :X/sst) :attrs :uniqueCount] (str (count @str->ref))))))
