(ns navaratine.xlsx-test
  (:require [navaratine.xlsx :as sut]
            [clojure.java.io :as io]
            [clojure.string :as string]
            [clojure.test :as t]
            [com.rpl.specter :as sr]
            [navaratine.xlsx.utils :as u]
            [navaratine.xlsx.shared-strings :as shared-strings])
  (:import (java.util.zip ZipInputStream ZipOutputStream)
           (java.io ByteArrayOutputStream)))


(defn- list-zipped-files
  [zis]
  (loop [acc #{}
         zen (.getNextEntry zis)]
    (if zen
      (recur (conj acc (.getName zen))
             (do (.closeEntry zis) (.getNextEntry zis)))
      acc)))


(def ss-mapping
  {"0"  "First name"
   "1"  "Last name"
   "10" "Example"
   "2"  "Group"
   "3"  "Presence"
   "4"  "Food"
   "5"  "Other"
   "6"  "Total"
   "7"  "Paid"
   "8"  "Jon"
   "9"  "Snow"})


(defn- cells
  "For nice test output with grouping by cell index/"
  [sheet]
  (->> sheet
       (sr/transform
        [(u/tag= :X/worksheet) u/children
         (u/tag= :X/sheetData) u/children
         (u/tag= :X/row) u/children
         (u/tag= :X/c) u/children
         (u/tag= :X/v) :content]
        #(map str %))
       (sr/select
        [(u/tag= :X/worksheet) u/children
         (u/tag= :X/sheetData) u/children
         (u/tag= :X/row) u/children
         (u/tag= :X/c)])
       (group-by #(get-in % [:attrs :r]))))


(t/deftest load-and-partially-render-test
  (with-open [outs  (ByteArrayOutputStream.)
              zouts (ZipOutputStream. outs)]
    (let [parsed (#'sut/load-and-partially-render "data/template.xlsx" zouts)]
      (t/testing "files required for substitution are parsed"
                 (t/is (seq (:shared-strings parsed)))
                 (t/is (= (#'sut/parse-entry! (io/resource "data/template/sharedStrings.xml"))
                          (:shared-strings parsed)))
                 (t/is (seq (:workbook parsed)))
                 (t/is (= (#'sut/parse-entry! (io/resource "data/template/workbook.xml"))
                          (:workbook parsed)))
                 (t/is (seq (get-in parsed [:sheets "xl/worksheets/sheet1.xml"])))
                 (t/is (= (#'sut/parse-entry! (io/resource "data/template/sheet1.xml"))
                          (get-in parsed [:sheets "xl/worksheets/sheet1.xml"]))))
      (t/testing "other files are copied to ZipOutputStream"
                 (t/is (= #{"[Content_Types].xml"
                            "_rels/.rels"
                            "xl/_rels/workbook.xml.rels"
                            "xl/charts/chart1.xml"
                            "xl/drawings/_rels/drawing1.xml.rels"
                            "xl/drawings/drawing1.xml"
                            "xl/media/image1.png"
                            "xl/styles.xml"
                            "xl/tables/table1.xml"
                            "xl/theme/theme1.xml"
                            "xl/workbook.xml"
                            "xl/worksheets/_rels/sheet1.xml.rels"}
                          (-> outs
                              .toByteArray
                              io/input-stream
                              ZipInputStream.
                              list-zipped-files)))))))


(t/deftest transform-sheet-test
  (t/testing "without replacements"
             (let [decoder        (shared-strings/decoder ss-mapping)
                   [encoder enc*] (shared-strings/encoder)
                   [counter cnt*] (shared-strings/counter)]
               (t/is (= (-> (io/resource "data/template/sheet1.xml")
                            (#'sut/parse-entry!)
                            (cells))
                        (-> (io/resource "data/template/sheet1.xml")
                            (#'sut/parse-entry!)
                            (#'sut/transform-sheet
                             {}
                             decoder
                             encoder
                             counter)
                            (cells))))
               (t/is (= 12 @cnt*))
               (t/is (= 11 (count @enc*)))))
  (t/testing "with replacements"
             (let [decoder        (shared-strings/decoder ss-mapping)
                   [encoder enc*] (shared-strings/encoder)
                   [counter cnt*] (shared-strings/counter)]
               (t/is (= (-> (io/resource "data/rendered/sheet1_rendered.xml")
                            (#'sut/parse-entry!)
                            (cells))
                        (-> (io/resource "data/template/sheet1.xml")
                            (#'sut/parse-entry!)
                            (#'sut/transform-sheet
                             {3 (repeat 20 [nil string/upper-case "GG" 125.33])}
                             decoder
                             encoder
                             counter)
                            (cells))))
               (t/is (= 69 @cnt*))
               (t/is (= 11 (count @enc*))))))


(t/deftest render-stream-test
  #_(sut/render-file
     "test.zip"
     "data/template.xlsx"
     {"Rachunki" {3 (repeat 20 [nil string/upper-case "GG" 125.33])}})
  (t/testing "all files are present in final stream"
             (t/is (= #{"[Content_Types].xml"
                        "_rels/.rels"
                        "xl/_rels/workbook.xml.rels"
                        "xl/charts/chart1.xml"
                        "xl/drawings/_rels/drawing1.xml.rels"
                        "xl/drawings/drawing1.xml"
                        "xl/media/image1.png"
                        "xl/sharedStrings.xml"
                        "xl/styles.xml"
                        "xl/tables/table1.xml"
                        "xl/theme/theme1.xml"
                        "xl/workbook.xml"
                        "xl/worksheets/_rels/sheet1.xml.rels"
                        "xl/worksheets/sheet1.xml"}
                      (-> (sut/render-stream "data/template.xlsx" {})
                          .toByteArray
                          io/input-stream
                          ZipInputStream.
                          list-zipped-files)))))
