(ns leiningen.new.nedap-lib
  "Generate a library project."
  (:require
   [clojure.string :as string]
   [clojure.java.io :as io]
   [leiningen.new.templates :refer [renderer year date project-name ->files sanitize-ns name-to-path multi-segment]]
   [leiningen.core.main :as main])
  (:import
   (java.io File Reader)
   (java.util Properties)))

(def properties-map
  (delay
   (let [filename "nedap.lein-template.properties"
         file-exists-at-root? (-> filename File. .exists)]
     (when (or file-exists-at-root?
               (-> filename io/resource))
       (with-open [^Reader reader (io/reader (if file-exists-at-root?
                                               filename
                                               (io/resource filename)))]
         (doto (Properties.)
           (.load reader)))))))

(defn get-system-property [k]
  (or (get @properties-map k)
      (System/getProperty k)))

(defn gather [{:keys [prompt iteration-prompt parse-fn]}]
  (println prompt)
  (loop []
    (some-> iteration-prompt println)
    (if-some [choice (parse-fn (read-line))]
      (do
        (println)
        choice)
      (recur))))

(defn ask [prompt default]
  (gather {:prompt   (str prompt
                          (when default
                            (str "\n(default: " default ")")))
           :parse-fn (fn [v]
                       (cond
                         (string/blank? v) (do
                                             (when-not default
                                               (println "Please enter a value."))
                                             default)
                         true              v))}))

(defn confirmation [prompt]
  (gather {:prompt           prompt
           :iteration-prompt "(respond with 'yes' or 'no')"
           :parse-fn         (fn [v]
                               (cond
                                 (#{"y" "yes" "Y" "YES" "Yes"} v) true
                                 (#{"n" "no" "N" "NO" "No"} v)    false))}))

(defn env-truth [k]
  (let [truthy (fn [x]
                 (not (-> #{nil
                            'null
                            ""
                            false
                            'n
                            'no
                            'off
                            0}
                          (contains? x))))]
    (some-> k get-system-property .toLowerCase (string/replace #"( |'|\")" "") read-string truthy)))

(defn nedap-lib [raw-name]
  (let [default-github-username "nedap"
        github-username (or (get-system-property "nedap.lein-template.github_username")
                            (ask "Github user/organisation that will own the repo:"
                                 default-github-username))
        nedap? (= github-username
                  default-github-username)
        copyright-holder (or (get-system-property "nedap.lein-template.copyright_holder")
                             (ask "Copyright holder (example: My Company):"
                                  (when nedap?
                                    "Nedap")))
        group-id (or (get-system-property "nedap.lein-template.group_id")
                     (ask "Maven group id (example: com.mycompany):"
                          (when nedap?
                            "com.nedap.staffing-solutions")))
        prefix (or (get-system-property "nedap.lein-template.lib_prefix")
                   (ask "Clojure namespace prefix:"
                        (if nedap?
                          (str "nedap." raw-name)
                          (str group-id "." raw-name))))
        gpg-key (or (get-system-property "nedap.lein-template.gpg_email")
                    (if nedap?
                      "releases-staffingsolutions@nedap.com"
                      (ask "Email for GPG-signing releases:"
                           nil)))
        system? (if-some [choice (env-truth "nedap.lein-template.use_system")]
                  choice
                  (confirmation "Setup a com.stuartsierra/component system?"))
        nvd? (if-some [choice (env-truth "nedap.lein-template.use_nvd")]
               choice
               (confirmation "Setup lein-nvd?"))
        clojurescript? (if-some [choice (env-truth "nedap.lein-template.use_cljs")]
                         choice
                         (confirmation "Also setup ClojureScript?"))
        clojars? (if-some [choice (env-truth "nedap.lein-template.use_clojars")]
                   choice
                   (confirmation "Deploy to Clojars?"))
        license? (if-some [choice (env-truth "nedap.lein-template.do_license")]
                   choice
                   (confirmation "Publicly license?"))
        render (renderer "nedap-lib")
        escaped-prefix (string/replace prefix "." "\\.")
        main-ns (str prefix ".api")
        system-ns (str prefix ".system")
        data {:clojars?         clojars?
              :clojurescript?   clojurescript?
              :copyright-holder copyright-holder
              :date             (date)
              :escaped-prefix   escaped-prefix
              :github-username  github-username
              :gpg-key          gpg-key
              :group-id         group-id
              :initial-release  "unreleased"
              :license?         license?
              :name             (project-name raw-name)
              :namespace        main-ns
              :nvd?             nvd?
              :system-ns        system-ns
              :nedap?           nedap?
              :system?          system?
              :system-path      (name-to-path system-ns)
              :nested-dirs      (name-to-path main-ns)
              :prefix           prefix
              :prefix-dir       (name-to-path prefix)
              :raw-name         raw-name
              :year             (year)}]
    (->> [["project.clj" (render "project.mustache" data)]
          ["README.md" (render "README.md" data)]
          [".gitignore" (render "gitignore" data)]
          ["nvd_suppressions.xml" (render "nvd_suppressions.xml" data)]
          [".github/pull_request_template.md" (render "pull_request_template.md" data)]
          [".github/PULL_REQUEST_TEMPLATE/ncrw.md" (render "ncrw_pull_request_template.md" data)]
          [".github/ISSUE_TEMPLATE/bug.md" (render "bug.md" data)]
          [".github/ISSUE_TEMPLATE/feature.md" (render "feature.md" data)]
          [".github/ISSUE_TEMPLATE/technical_improvement.md" (render "technical_improvement.md" data)]
          ["dev/dev.clj" (render "dev.clj" data)]
          (when system?
            [(str "src/{{system-path}}.clj" (when clojurescript?
                                              "c"))
             (render "system.clj" data)])
          [(str "src/{{nested-dirs}}.clj" (when clojurescript?
                                            "c"))
           (render "api.cljc" data)]
          [(str "test/unit/{{nested-dirs}}.clj" (when clojurescript?
                                                  "c"))
           (render "test.cljc" data)]
          (when clojurescript?
            ["test/{{prefix-dir}}/test_runner.cljs" (render "test_runner.cljs" data)])
          (when license?
            ["LICENSE" (render "LICENSE" data)])
          [".circleci/config.yml" (render "config_yml.mustache" data)]
          "resources"]
         (filter some?)
         (apply ->files data))))
