(ns {{prefix}}.assets.component
  (:require
   [cheshire.core :as cheshire]
   [clojure.java.io :as io]
   [com.stuartsierra.component :as component]
   [{{prefix}}.assets.kws :as assets]
   [{{prefix}}.assets.protocols.assets :as protocols.assets]
   [nedap.components.assets.component :as assets.component]
   [nedap.speced.def :as speced]
   [nedap.utils.io.api :refer [running-in-jar?]]
   [nedap.utils.modular.api :refer [implement]]
   [stefon.core :as stefon]
   [stefon.manifest]))

(def stylesheet-name "css/stylesheet.css")

(defn ensure-path [path]
  (if (running-in-jar?)
    path
    (str "resources/" path)))

(def ^{:doc     "Maps webjar paths to desired asset filenames."
       :example {"META-INF/resources/webjars/tether/1.4.0/js/tether.js" "javascript/webjars/tether.js"}}
  webjars-mappings
  {})

(def stylesheets [stylesheet-name])

(def javascripts [])

(def manifest-file "manifest.json")

(def stefon-options
  {:asset-roots   [(ensure-path "assets")]
   :serving-root  (ensure-path "public")
   :mode          :production
   :manifest-file (if (running-in-jar?) ;; battle-tested workaround
                    (-> manifest-file io/resource str)
                    (-> manifest-file ensure-path))
   :precompiles   (->> webjars-mappings
                       vals
                       (into stylesheets)
                       (into javascripts))})

(defn new-assets-compiler [{:keys [stylesheet]}]
  {:pre [stylesheet]}
  (assets.component/new {::assets.component/garden-options {:id           "app"
                                                            :source-paths ["src"]
                                                            :stylesheet   stylesheet
                                                            :compiler     {:pretty-print? true
                                                                           :output-to     (str "resources/assets/" stylesheet-name)}}
                         ::assets.component/stefon-options stefon-options
                         ::assets.component/webjar-options {::assets.component/webjars.mappings        webjars-mappings
                                                            ::assets.component/webjars.asset-directory "resources/assets/"}}))

(speced/defn link-to-asset [this, ^string? asset-name]
  (stefon/link-to-asset asset-name stefon-options))

(defn start [this]
  ;; make manifest.json not choke on the first run
  (alter-var-root #'stefon.manifest/load! (constantly (fn []
                                                        (some-> manifest-file
                                                                io/resource
                                                                slurp
                                                                cheshire/parse-string
                                                                stefon.manifest/load-map!))))

  (stefon/init stefon-options)
  (implement {::assets/javascripts javascripts
              ::assets/stylesheets stylesheets}
    protocols.assets/--link-to-asset link-to-asset))

(defn stop [this]
  {})

(defn new []
  (implement {}
    component/start start
    component/stop  stop))
