(ns kitashiro.ajax
  (:require [ajax.core :as cljs-ajax]
            [cljs-time.core :as time]
            [cljs.core.async :as async :refer [>! <! chan close! put!]]))


;; https://github.com/JulianBirch/cljs-ajax/blob/master/src/ajax/core.cljs
;; copy of the default json formatter, but returns a map with json body
;; in :resp and extra request metadata: :response-headers, :url, :method, and :request-time
(defn json-response-format
  "Returns a JSON response format. Options include
  :keywords? Returns the keys as keywords
  :prefix A prefix that needs to be stripped off. This is to
  combat JSON hijacking. If you're using JSON with GET request,
  you should use this.
  http://stackoverflow.com/questions/2669690/why-does-google-prepend-while1-to-their-json-responses
  http://haacked.com/archive/2009/06/24/json-hijacking.aspx"
  ([{:keys [prefix keywords? url method start-time]
     :or {start-time (time/now)}}]
   {:read (fn read-json [xhrio]
            (let [json (js/JSON.parse (.getResponseText xhrio))
                  headers (js->clj (.getResponseHeaders xhrio))
                  request-time (try
                                 (time/in-millis (time/interval start-time (time/now)))
                                 (catch :default e
                                   (.apply (.-error js/console) js/console (clj->js e))
                                   0))]
              {:resp (js->clj json :keywordize-keys keywords?)
               :response-headers headers
               :url url
               :method method
               :request-time request-time}))
    :description (str "JSON"
                      (if prefix (str " prefix '" prefix "'"))
                      (if keywords? " keywordize"))}))


(defn xml-request-format []
  {:content-type "application/xml"
   :write identity})

(defn xml-response-format []
  {:read (fn read-xml [xhrio]
           {:resp (.getResponseXml xhrio)})
   :description "XML"})

(defn raw-response-format []
  {:read (fn read-text [xhrio]
           {:resp (.getResponseText xhrio)})})


(defn normalize-error-response [default-response props]
  (-> default-response
      (merge props)
      (assoc :status-code (:status default-response))
      (assoc :resp (get-in default-response [:response :resp]))
      (assoc :status :failed)))


(defn ajax-opts [{:keys [keywords? context headers format uri method]
                  :or {keywords? true format :json}
                  :as opts}]
  (let [format-opts (case format
                      :json {:format (cljs-ajax/json-request-format)
                             :response-format (json-response-format {:keywords? keywords? :url uri :method method})
                             :keywords? keywords?
                             :headers (merge {:Accept "application/json"}
                                             headers)}
                      :xml {:format (xml-request-format)
                            :response-format (xml-response-format)
                            :headers (merge {:Accept "application/xml"}
                                            headers)}
                      :raw {:format :raw
                            :response-format (raw-response-format)
                            :headers headers})]
    (cljs-ajax/transform-opts (merge opts format-opts))))


(defn managed-ajax [method url & {:as opts}]
  (let [channel (chan)
        base-opts {:method method
                   :url url
                   :handler #(put! channel (assoc % :status :success))
                   :error-handler #(put! channel (normalize-error-response % {:url url}))
                   :finally #(close! channel)}]
    (-> (merge base-opts opts)
        ajax-opts
        cljs-ajax/ajax-request)
    channel))

