(ns kitashiro.dom
  (:use-macros [purnam.core :only [! !>]]
               [cljs.core.async.macros :only [go]])
  (:require [cljs.core.async :refer [<! >! chan]]
            [goog.events :as events]
            [goog.fx.dom :as fx-dom]
            [goog.fx.Animation.EventType :as EventType]
            [clojure.string :as string]))


(defn text [elem]
  (or (.-textContent elem) (.-innerText elem)))


(defn set-html!
  "Set the innerHTML of `elem` to `html`"
  [elem html]
  (set! (.-innerHTML elem) html)
  elem)


(defn sel1
  ([selector]        (sel1 js/document selector))
  ([parent selector] (.querySelector parent selector)))


(defn sel
  ([selector]        (sel js/document selector))
  ([parent selector] (.querySelectorAll parent selector)))


(defn fade-out
  "fade out the selected node. return channel."
  [el msec]
  (let [ch (chan)]
    (-> (fx-dom/FadeOut. el msec)
        (events/listen EventType/END #(go (>! ch el)))
        (.play))
    ch))


(defn fade-in
  "fade in the selected node. return channel."
  [el msec]
  (let [ch (chan)]
    (-> (fx-dom/FadeIn. el msec)
        (events/listen EventType/END #(go (>! ch el)))
        (.play))
    ch))


(defn emulate-parse-html
  "Emulate DOMParser for Safari. https://developer.mozilla.org/en-US/docs/Web/API/DOMParser"
  [html-str]
  (let [doc (!> js/document.implementation.createHTMLDocument "")] ;; https://bitbucket.org/apribase/kitashiro/issue/2/documentimplementationcreatehtmldocument
    (if (->> (string/lower-case html-str)
             (re-find #"<!doctype ([^>]*)>")
             (second))
      (! doc.documentElement.innerHTML html-str)
      (! doc.body.innerHTML html-str))
    doc))


(defn parse-html
  "parse the html string to HTMLDocument."
  [html-str]
  (if-let [result (-> (js/DOMParser.)
                      (.parseFromString html-str "text/html"))]
    result
    (emulate-parse-html html-str)))
