(ns kitashiro.spa.ajax
  (:refer-clojure :exclude [get])
  (:require [cljs.core.async :as async :refer [chan put!]]
            [ajax.core :as ajax]
            [cljs.reader :as edn]))

;; cljs-http は query に {:accept "application/edn"} をセットすると
;; HTTP Headers に Accept を追加できる。
;; しかし送られてきた octetstream を EDN として処理する機能はないようだ。

;; cljs-ajax は query に {:response-format (ajax-edn/response-format-edn)} をセットすると
;; HTTP Headers に Accept を追加して、さらに octetstream を EDN として処理する。
;; 詳細情報も取れる cljs-http を使いたいところだが、上記の EDN の扱いのために cljs-ajax を使用する。


;; cljs-http のレスポンスは以下のようになるので、cljs-ajax でもなるべく合わせる。
;; {:status 200
;;  :success true
;;  :body ""
;;  :headers nil
;;  :trace-redirects nil
;;  :error-code :no-error
;;  :error-text nil}

(defn normalize-success-response
  [response]
  {:success true
   :body response})


(defn normalize-error-response
  " default response: https://github.com/JulianBirch/cljs-ajax#error-responses"
  [response]
  (merge {:success false}
         response))


(defn get [url & [query]]
  (let [ch (chan)]
    (ajax/GET url (merge query {:handler #(put! ch (normalize-success-response %))
                                :error-handler #(put! ch (normalize-error-response %))}))
    ch))



;; Fetch API
(defn fetch [url & [query]]
  (let [ch (chan)]
    (-> (js/fetch url (clj->js query))
        (.then (fn [response]
                 (if (.-ok response)
                   (condp = (:response-format query)
                     :json (-> (.json response)
                               (.then (fn [json]
                                        (put! ch {:success true
                                                  :body json
                                                  :raw response}))))
                     :edn (-> (.text response)
                              (.then (fn [text]
                                       (put! ch {:success true
                                                 :body (edn/read-string text)
                                                 :raw response}))))
                     (-> (.text response) ;; default is text format
                         (.then (fn [text]
                                  (put! ch {:success true
                                            :body text
                                            :raw response})))))
                   (put! ch {:success false
                             :body response
                             :raw response})))))
    ch))
