(ns kitashiro.spa.history
  (:require [clojure.string :as string]
            [goog.events :as events]
            [secretary.core :as sec])
  (:import [goog.dom DomHelper]
           [goog.events EventType BrowserEvent]
           [goog.history Html5History]))


(defn path-matches?
  "True if the two tokens are the same except for the fragment"
  [token-a token-b]
  (= (first (string/split token-a #"#"))
     (first (string/split token-b #"#"))))


(defn new-window-click?
  [^Event event]
  (or (.isButton event goog.events.BrowserEvent.MouseButton.MIDDLE)
      (and (.-platformModifierKey event)
           (.isButton event goog.events.BrowserEvent.MouseButton.LEFT))))


;; see this.transformer_ at http://goo.gl/ZHLdwa
(def ^{:doc "Custom token transformer that preserves hashes"}
  token-transformer
  (let [transformer (js/Object.)]
    (set! (.-retrieveToken transformer)
          (fn [path-prefix location]
            (str (subs (.-pathname location) (count path-prefix))
                 (when-let [query (.-search location)]
                   query)
                 (when-let [hash (second (string/split (.-href location) #"#"))]
                   (str "#" hash)))))
    (set! (.-createUrl transformer)
          (fn [token path-prefix _location]
            (str path-prefix token)))
    transformer))


(defn set-current-token!
  "Lets us keep track of the history state, so that we don't dispatch twice on the same URL"
  [^Html5History history-imp]
  (set! (.-_current_token history-imp) (.getToken history-imp)))


(defn setup-dispatcher!
  [^Html5History history-imp]
  (events/listen history-imp goog.history.EventType.NAVIGATE
                 (fn [^Html5History h]
                   (set-current-token! history-imp)
                   (sec/dispatch! (str "/" (.-token h))))))


(defn bootstrap-dispatcher!
  "We need lots of control over when we start listening to navigation events because
  we may want to ignore the first event if the server sends an error status code (e.g. 401)
  This function lets us ignore the first event that history-imp fires when we enable it. We'll
  manually dispatch if there is no error code from the server."
  [^Html5History history-imp]
  (events/listenOnce history-imp goog.history.EventType.NAVIGATE #(setup-dispatcher! history-imp)))


(defn disable-erroneous-popstate!
  "Stops the browser's popstate from triggering NAVIGATION events unless the url has really
  changed. Fixes duplicate dispatch in Safari and the build machines."
  [^Html5History history-imp]
  ;; get this history instance's version of window, might make for easier testing later
  (let [window (.-window_ history-imp)]
    (events/removeAll window goog.events.EventType.POPSTATE)
    (events/listen window goog.events.EventType.POPSTATE
                   #(if (= (.getToken history-imp)
                           (.-_current_token history-imp))
                      (js/console.log (str "Ignoring duplicate dispatch event to" (.getToken history-imp)))
                      (.onHistoryEvent_ history-imp %)))))


(defn closest-tag [element query-tag]
  "Handle SVG element properly."
  (let [dom-helper (goog.dom.DomHelper.)
        upper-query (string/upper-case query-tag)
        lower-query (string/lower-case query-tag)]
    (or (when (= (.-tagName element) upper-query) element)
        (.getAncestorByTagNameAndClass dom-helper element upper-query)
        (when (instance? js/SVGElement element)
          (loop [e element]
            (let [e-tag (.-tagName e)]  ; SVG tags are case sensitive
              (condp = e-tag
                lower-query e
                "svg" nil
                (recur (.-parentElement e)))))))))

(defn closest-a-tag [target]
  (closest-tag target "A"))


(defn navigate
  [^Html5History history-imp new-token]
  (if (path-matches? (.getToken history-imp) new-token) ;; reload or scroll to hash
    (.replaceToken history-imp new-token)
    (.setToken history-imp new-token)))


(def history-imp
  ;; need a history element, or goog will overwrite the entire dom
  (doto (goog.history.Html5History. js/window token-transformer)
    (.setUseFragment false)
    (.setPathPrefix "/")
    (bootstrap-dispatcher!)
    (set-current-token!) ; Stop Safari from double-dispatching
    (disable-erroneous-popstate!) ; Stop Safari from double-dispatching
    (.setEnabled true)))


(defn setup-link-dispatcher!
  [top-level-node]
  (events/listen top-level-node goog.events.EventType.CLICK
                 #(let [-target   (.-target %)
                        a         (closest-a-tag -target)
                        location  (when a (str (.-pathname a) (.-search a) (.-hash a)))
                        new-token (when (seq location) (subs location 1))]
                    (when (and (seq location) ;; in my page navigation
                               (= js/window.location.hostname (.-hostname a))
                               (not (or (new-window-click? %) (= (.-target a) "_blank")))
                               (not= (-> a (aget "dataset") (aget "secretary")) "disable")) ;; you can disable secretary by data-secretary disable.
                      (.stopPropagation %)
                      (.preventDefault  %)
                      (navigate history-imp new-token)))))


(defn setup-input-dispatcher!
  [input-el search-root]
  (events/listen input-el goog.events.EventType.KEYDOWN
                 #(when (= 13 (.-keyCode %)) ;; ENTER
                    (.preventDefault %)
                    (.blur input-el)
                    (navigate history-imp (str search-root (js/encodeURIComponent (.-value input-el)) "/")))))





