(ns net.cgrand.seqexp.parser)

(defn- link
  "(idempotent)"
  [instructions]
  (let [[insts labels] (reduce (fn [[insts labels] [op arg :as inst]]
                               (if (= :label op)
                                 [insts (assoc labels arg (count insts))]
                                 [(conj insts inst) labels]))
                         [[] {}] instructions)]
    (mapv (fn [[op arg :as inst] pc]
            (case op
              (:fork :jump) [op (if-let [dest (labels arg)] (- dest pc) arg)]
              :call [op (labels arg arg)]
              inst))
      insts
      (range))))

(defmacro ^:private asm [& exprs]
  (let [gen (gensym 'gen)
        exprs (partition 2 exprs)]
    `(let [~gen (memoize gensym)]
       (concat
         ~@(map
             (fn [[op arg]]
               (case op
                 (label jump fork call) [[(keyword op) (list gen (keyword arg))]]
                 include `(instructions ~arg)
                 (pred save return) [[(keyword op) arg]]))
             exprs)))))

;; a "stack" is a map whose values are stacks themselves.
;; {}, {1 {}}, {2 {} 1 {}}, {3 {2 {} 1 {}}}

(defn- merge-stacks [a b]
  (merge-with merge-stacks a b))

(defn add-thread [threads pc stack insts]
  (let [[inst arg] (nth insts pc nil)]
    (case inst
      (:pred nil) (merge-stacks threads {pc stack})
      :jump (recur threads (+ pc arg) stack insts)
      :fork (-> threads
              (add-thread (inc pc) stack insts)
              (add-thread (+ pc arg) stack insts))
      :call (recur threads arg {(inc pc) stack} insts)
      :return (reduce-kv 
                (fn [threads pc stack]
                  (add-thread threads pc stack insts))
                threads stack))))

(defn stepper [insts]
  (fn [threads c]
    (reduce-kv (fn [threads pc stack]
                 (if-let [[_ pred] (nth insts pc nil)]
                   (if (pred c) 
                     (add-thread threads (inc pc) stack insts)
                     threads)
                   threads))
      {} threads)))

(def lisp
  (link
    (asm
      call E
      jump end
      label sym
      pred #(<= (long \a) (long %) (long \z ))
      fork sym
      return nil
      label E
      call sym
      fork E
      label eend
      return nil
      label end
      )))